
// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Timothy J. McBrayer 
//          Malolan Chetlur     
//          Narayanan Thondugulam
//          Krishnan Subramani  
//          Magnus Danielson    cfmd@swipnet.se

#include "IIR_DesignatorList.hh"
#include "IIR_ComponentInstantiationStatement.hh"
#include "IIR_DesignatorExplicit.hh"
#include "IIR_Label.hh"
#include "IIR_SimpleName.hh"
#include "IIR_AboveAttribute.hh"
#include "IIR_TextLiteral.hh"
#include "error_func.hh"
#include "set.hh"
#include "published_file.hh"
#include "sstream-wrap.hh"

IIRScram_DesignatorList::~IIRScram_DesignatorList() {}

void 
IIRScram_DesignatorList::_publish_vhdl(ostream &_vhdl_out) {
  IIR_Designator* list_element;

  list_element = first();

  ASSERT ( list_element != NULL );
  
  list_element->_publish_vhdl(_vhdl_out);

  list_element = successor(list_element);
  while (list_element != NULL) {
    _vhdl_out << ", ";
    list_element->_publish_vhdl(_vhdl_out);
    list_element = successor(list_element);
  }
}

void
IIRScram_DesignatorList::_publish_cc_elements( published_file &_cc_out ) {
  IIR_Designator* element = first();

  while(element != NULL) {
    element->_publish_cc_designator( _cc_out );
    _cc_out << "_";
    element = successor(element);
  }
}

void
IIRScram_DesignatorList::_get_list_of_input_signals( set<IIR> *list ){
  IIR_Designator *element = first();
  while( element != NULL) {
    element->_get_list_of_input_signals(list);
    element = successor(element);
  }
}

void
IIRScram_DesignatorList::_add_decl_into_cgen_symbol_table() {
  IIR_Designator* element;
  element = first();
  while(element != NULL) {
    element->_add_decl_into_cgen_symbol_table();
    element = successor(element);
  }
}

// Modified by SK to copy the names from the argument to the list.  This
// may be better done by blindly copying the elements, but this involves
// some check.
void
IIRScram_DesignatorList::_add_signals_to_sensitivity_list(IIR_DesignatorList* sensitivity_list) {
  IIR_Designator* sig_desig;
  sig_desig = sensitivity_list->first();
  while(sig_desig != NULL) {
    ASSERT(sig_desig->get_kind() == IIR_DESIGNATOR_EXPLICIT);
    ASSERT(sig_desig->_get_name() != NULL);
    this->append(sig_desig);
    sig_desig = sensitivity_list->successor(sig_desig);
  }
}

IIR_Label *
IIRScram_DesignatorList::_process_instantiate_statement( IIR_Declaration *instantiated_unit,
							 IIR *statement ){
  ASSERT( instantiated_unit != NULL );
  ASSERT( statement != NULL );

  IIR_Label *retval = NULL;

  if( statement->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT ){
    IIR_ComponentInstantiationStatement *as_instantiate_statement;
    as_instantiate_statement = (IIR_ComponentInstantiationStatement *)statement;
    ASSERT( as_instantiate_statement->get_instantiated_unit()->_is_iir_declaration() == TRUE );
    if( as_instantiate_statement->get_instantiated_unit() == instantiated_unit ){
      ASSERT( as_instantiate_statement->get_label() != NULL );
      ASSERT( as_instantiate_statement->get_label()->get_statement() != NULL );
      retval = as_instantiate_statement->get_label();
    }
  }

  return retval;
}

void 
IIRScram_DesignatorList::_process_explicit_list( IIR_Declaration *instantiated_unit,
						 IIR_List &statement_list,
						 IIR *configuration ){
  IIR_SimpleName *label_name = NULL;
  (void)instantiated_unit;  // quiet down the compiler

  IIR_Designator *current_designator = first();
  while( current_designator != NULL ){
    ASSERT( current_designator->get_kind() == IIR_DESIGNATOR_EXPLICIT );
    IIR_DesignatorExplicit *explicit_designator = (IIR_DesignatorExplicit *)current_designator;
    ASSERT( explicit_designator->get_name() != NULL );
    ASSERT( explicit_designator->get_name()->get_kind() == IIR_SIMPLE_NAME );
    label_name = (IIR_SimpleName *)explicit_designator->get_name();

    IIR_Label *instantiate_label = statement_list._find_instantiate_label( label_name );

    if( instantiate_label == NULL ){
      ostringstream err;
      err << "No label |" << *label_name << "| found.";
      report_error( label_name, err.str() );
      break;
    }
    else{
      explicit_designator->set_name( label_name->_decl_to_decl( instantiate_label ) );
      ASSERT( instantiate_label->get_statement()->get_kind() ==
	      IIR_COMPONENT_INSTANTIATION_STATEMENT );
      IIR_ComponentInstantiationStatement *statement;
      statement = (IIR_ComponentInstantiationStatement *)instantiate_label->get_statement();
      statement->_set_configuration( configuration );
    }
    current_designator = successor( current_designator );
  }
}

void 
IIRScram_DesignatorList::_process_by_all_or_others( IIR_Declaration *instantiated_unit,
						    IIR_List &statement_list,
						    IIR_Boolean by_all,
						    IIR *configuration ){
  IIR_DesignatorList new_designator_list;
  IIR_Label *new_label = NULL;

  IIR *current_statement = statement_list.first();
  while( current_statement != NULL ){
    new_label = _process_instantiate_statement( instantiated_unit, 
						current_statement );

    if( new_label != NULL ){
      ASSERT( new_label->get_statement()->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT );
      IIR_ComponentInstantiationStatement *statement;
      statement = (IIR_ComponentInstantiationStatement *)new_label->get_statement();
      if( by_all == TRUE || statement->_get_configuration() == NULL ){
	IIR_DesignatorExplicit *new_designator = new IIR_DesignatorExplicit();
	copy_location( new_label, new_designator );
	new_designator->set_name( new_label );
	statement->_set_configuration( configuration );
	new_designator_list.append( new_designator );
      }
    }
    current_statement = statement_list.successor( current_statement );
  }

  if( new_designator_list.num_elements() > 0 ){
    // Destroy the current list.
    ASSERT( num_elements() == 1 );
    IIR_Designator *current = first();
    remove( current );
    delete current;

    current = new_designator_list.first();
    while( current != NULL ){
      append( current );
      current = new_designator_list.successor( current );
    }

  }
}

void 
IIRScram_DesignatorList::_type_check_as_instantiation_list( IIR *instantiated_unit,
							    IIR_List &statement_list,
							    IIR *configuration ){
  ASSERT( instantiated_unit != NULL );
  ASSERT( instantiated_unit->_is_iir_declaration() == TRUE );
  IIR_Declaration *as_decl = (IIR_Declaration *)instantiated_unit;

  IIR_Designator *current_designator = first();
  if( current_designator != NULL ){
    switch( current_designator->get_kind() ){
    case IIR_DESIGNATOR_BY_ALL:
      _process_by_all_or_others( as_decl, statement_list, TRUE, configuration );
      break;
      
    case IIR_DESIGNATOR_BY_OTHERS:
      _process_by_all_or_others( as_decl, statement_list, FALSE, configuration );
      break;
      
    case IIR_DESIGNATOR_EXPLICIT:
      _process_explicit_list( as_decl, statement_list, configuration );    
      break;
      
    default:
      ostringstream err;
      err << "Internal error in IIRScram_DesignatorList::_type_check_as_instantiation_list "
	  << "- don't know what to do with node type " << current_designator->get_kind_text();
      report_error( current_designator, err.str() );
    }
  }

#ifdef DEVELOPER_ASSERTIONS
  current_designator = first();
  while( current_designator != NULL ){
    if( current_designator->get_kind() == IIR_DESIGNATOR_EXPLICIT ){
      IIR_DesignatorExplicit *as_explicit = 
	static_cast<IIR_DesignatorExplicit *>(current_designator);
      ASSERT( as_explicit->get_name() != NULL );
      ASSERT( as_explicit->get_name()->get_kind() == IIR_LABEL );
      IIR_Label *current_label = static_cast<IIR_Label *>(as_explicit->get_name());
      ASSERT( current_label->get_statement()->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT );
      ASSERT( static_cast<IIR_ComponentInstantiationStatement *>(current_label->get_statement())->_get_configuration() != NULL );
    }
    current_designator = successor( current_designator );
  }
#endif
  
}

IIR_Boolean
IIRScram_DesignatorList::_is_component_in_the_list(IIR_TextLiteral* component_name) {
  IIR_Int32 wanted_instantiation = -1;
  IIR_Designator* designator = this->first();
  while(designator != NULL) {
    switch(designator->get_kind()) {
    case IIR_DESIGNATOR_EXPLICIT:
      ASSERT(designator->_get_name()->_is_iir_declaration() == TRUE);
      wanted_instantiation = 
	IIR_TextLiteral::_cmp(((IIR_Declaration*)designator->_get_name())->get_declarator(), component_name);
      if(wanted_instantiation == 0) {
	return TRUE;
      }
      break;
    case IIR_DESIGNATOR_BY_ALL:
    case IIR_DESIGNATOR_BY_OTHERS:
      return TRUE;
      break;
      
    default:
      break;
    }	
    designator = this->successor(designator);
  }
  return FALSE;
}

IIR_Boolean
IIRScram_DesignatorList::_is_above_attribute_found() {
  IIR_Boolean retval = FALSE;
  IIR_Designator* element = first();
  IIR *current_name = NULL;
  while(element != NULL) {
    if(element->get_kind() == IIR_DESIGNATOR_EXPLICIT) {
      current_name = ((IIR_DesignatorExplicit*)element)->get_name();
      retval = retval || current_name->_is_above_attribute_found();
    }
    else {
      retval = retval || element->_is_above_attribute_found();
    }
    element = successor(element);
  }
  return retval;
}

void
IIRScram_DesignatorList::
_build_above_attribute_set(set<IIR_AboveAttribute> *to_build) {
 IIR_Designator* element = first();

  while(element != NULL) {
    element->_build_above_attribute_set(to_build);
    element = successor(element);
  }
}

visitor_return_type *IIRScram_DesignatorList::_accept_visitor(node_visitor *visitor, visitor_argument_type *arg) {
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_DesignatorList(this, arg);
};
