
// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Timothy J. McBrayer 
//          Malolan Chetlur     
//          Krishnan Subramani  
//          Radharamanan Radhakrishnan
//          Narayanan Thondugulam
//	    Magnus Danielson	cfmd@swipnet.se

#include "IIRScram.hh"
#include "IIR_AboveAttribute.hh"
#include "IIR_AssociationList.hh"
#include "IIR_AssociationElementByExpression.hh"
#include "IIR_AssociationElementOpen.hh"
#include "IIR_ComponentDeclaration.hh"
#include "IIR_FunctionCall.hh"
#include "IIR_FunctionDeclaration.hh"
#include "IIR_IndexedName.hh"
#include "IIR_InterfaceList.hh"
#include "IIR_SignalDeclaration.hh"
#include "IIR_SignalInterfaceDeclaration.hh"
#include "IIR_ComponentInstantiationStatement.hh"
#include "error_func.hh"
#include "resolution_func.hh"
#include "symbol_table.hh"
#include "set.hh"
#include "published_file.hh"
#include "sstream-wrap.hh"

IIRScram_AssociationList::~IIRScram_AssociationList() {}

void 
IIRScram_AssociationList::append( IIR_AssociationElement *to_append ){
  if( to_append != NULL ){
    ASSERT( to_append->_is_association() == TRUE );
  }

  dl_list<IIR>::append( to_append );
}

void 
IIRScram_AssociationList::prepend( IIR_AssociationElement *to_prepend ){
  if( to_prepend != NULL ){
    ASSERT( to_prepend->_is_association() == TRUE );
  }

  dl_list<IIR>::prepend( to_prepend );
}

IIR_AssociationElement *
IIRScram_AssociationList::first(){
  IIR *retval = dl_list<IIR>::first();
  
  if( retval != NULL ){
    ASSERT( retval->_is_association() == TRUE );
  }

  return (IIR_AssociationElement *)retval;
}

IIR_AssociationElement *
IIRScram_AssociationList::successor( IIR_AssociationElement *to_succeed ){
  IIR *retval = dl_list<IIR>::successor( to_succeed );

  if( retval != NULL ){
    ASSERT( retval->_is_association() == TRUE );
  }

  return (IIR_AssociationElement *)retval;
}

IIR_AssociationElement *
IIRScram_AssociationList::get_nth_element( int to_get ){
  IIR *retval = dl_list<IIR>::get_nth_element( to_get );
  if( retval != NULL ){
    ASSERT( retval->_is_association() == TRUE );
  }
  
  return (IIR_AssociationElement *)retval;
}

void 
IIRScram_AssociationList::_replace( IIR_AssociationElement *to_replace,
				    IIR_AssociationElement *replace_with ){
  ASSERT( to_replace->_is_association() == TRUE );
  ASSERT( replace_with->_is_association() == TRUE );

  dl_list<IIR>::_replace( to_replace, replace_with );
}

void 
IIRScram_AssociationList::_publish_vhdl(ostream &_vhdl_out) {
  IIR_AssociationElement *node = first();

  if (node != NULL) {
    node->_publish_vhdl(_vhdl_out);
    for (node = successor(node); node != NULL; node = successor(node)) {
      _vhdl_out << ", ";
      node->_publish_vhdl(_vhdl_out);
    }
  }
}

void
IIRScram_AssociationList::_publish_vhdl_without_formals(ostream &_vhdl_out) {
  IIR_AssociationElement *node = first();

  if (node != NULL) {
    node->_publish_vhdl_without_formals(_vhdl_out);
    for (node = successor(node); node != NULL; node = successor(node)) {
      _vhdl_out << ", ";
      node->_publish_vhdl_without_formals(_vhdl_out);
    }
  }
}

void
IIRScram_AssociationList::_publish_cc_for_index( published_file &_cc_out )
{
  IIR_AssociationElement *node = first();

  SCRAM_CC_REF( _cc_out, "IIRScram_AssociationList::_publish_cc_for_index" );
  
  if (node != NULL) {
    node->_publish_cc_lvalue( _cc_out );
    for (node = successor(node); node != NULL; node = successor(node)) {
      _cc_out << "][";
      node->_publish_cc_lvalue( _cc_out );
    }
  }
}

void 
IIRScram_AssociationList::_publish_cc_lvalue( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_AssociationList::_publish_cc" );

  IIR_AssociationElement *node = first();
  while( node != 0 ){
    node->_publish_cc_lvalue( _cc_out );
    node = successor( node );
    if( node != 0 ){
      _cc_out << "," << NL();
    }
  }
}

void 
IIRScram_AssociationList::_publish_cc_generic_map_aspect( published_file &_cc_out ) {
  IIR_AssociationElement *node = first();

  SCRAM_CC_REF( _cc_out, "IIRScram_AssociationList::_publish_cc_generic_map_aspect" );

  if (node != NULL) {
    node->_publish_cc_generic_map_aspect( _cc_out );
    for (node = successor(node); node != NULL; node = successor(node)) {
      _cc_out << "," << NL();
      node->_publish_cc_generic_map_aspect( _cc_out );
    }
  }
}

void
IIRScram_AssociationList::_publish_cc_generic_map_aspect_for_conc_stmts( published_file &_cc_out )
{
  IIR_AssociationElement *node = first();
  PublishedUnit tmpUnit = _get_currently_publishing_unit();
  _set_currently_publishing_unit(GENERIC_MAP);

  SCRAM_CC_REF( _cc_out, "IIRScram_AssociationList::_publish_cc_generic_map_aspect_for_conc_stmts" );

  while (node != NULL) {
    node->_publish_cc_generic_map_aspect_for_conc_stmts( _cc_out );
    node = successor(node);

    if (node != NULL) {
      _cc_out << "," << NL();
    }
  }
  _set_currently_publishing_unit(tmpUnit);
}

void 
IIRScram_AssociationList::_publish_cc_bounds( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_AssociationList::_publish_cc_bounds" );
  _cc_out << "defaultInfo" ;
}

void 
IIRScram_AssociationList::_publish_cc_state_object_init( published_file &_cc_out ) {
  IIR_AssociationElement *node = first();

  SCRAM_CC_REF( _cc_out, "IIRScram_AssociationList::_publish_cc_state_object_init" );

  if (node != NULL) {
    node->_publish_cc_state_object_init( _cc_out );
    _set_aggregate_iterator_counter( _get_aggregate_iterator_counter() + 1);
    for (node = successor(node); node != NULL; node = successor(node)) {
      _cc_out << "," << NL();
      node->_publish_cc_state_object_init( _cc_out );
      _set_aggregate_iterator_counter( _get_aggregate_iterator_counter() + 1);
    }
  }
}

void 
IIRScram_AssociationList::_publish_cc_subprogram_arguments( published_file &_cc_out ) {
  IIR_AssociationElement *node = first();

  SCRAM_CC_REF( _cc_out, "IIRScram_AssociationList::_publish_cc_subprogram_arguments" );

  if (node != NULL) {
    node->_publish_cc_subprogram_arguments( _cc_out );
    for (node = successor(node); node != NULL; node = successor(node)) {
      _cc_out << "," << NL();
      node->_publish_cc_subprogram_arguments( _cc_out );
    }
  }
}

IIR_Int32
IIRScram_AssociationList::_get_number_of_element_associations() {
  IIR_Int32 number_of_elements = 0;
  IIR_AssociationElement *node = first();
  for (; node != NULL; node = successor(node)) {
    number_of_elements += node->_get_number_of_element_associations();
  }
  return number_of_elements;
}

void
IIRScram_AssociationList::_get_list_of_input_signals( set<IIR>* list ){
  IIR_AssociationElement *node = first();
  while( node != NULL ){
    node->_get_list_of_input_signals(list);
    node = successor(node);
  }
}

void
IIRScram_AssociationList::_get_signal_source_info( set<IIR> *siginfo ){
  IIR_AssociationElement *node = first();
  while(node != NULL) {
    node->_get_signal_source_info(siginfo);
    node = successor(node);
  }
}

void
IIRScram_AssociationList::_publish_cc_port_map_associations( published_file &_cc_out ) {
  IIR_AssociationElement *association = NULL;
  IIR *formal = NULL;
  IIR *interface_decl = NULL;

  SCRAM_CC_REF( _cc_out, "IIRScram_AssociationList::_publish_cc_port_map_associations" );

  association = first();
  formal = association->get_formal();
  ASSERT(formal != NULL);
  if(formal != NULL) {
    // ### assumes the parameter in order, also on by expression not open yet
    while( association != NULL ) {
      interface_decl = association->get_formal();
      _cc_out << OS("tmpPortMap->addPortAssociation(") << "&component->";
      association->_get_actual()->_publish_cc_elaborate( _cc_out );
      _cc_out << "," << NL();
      _cc_out << "&architecture->";
      association->get_formal()->_publish_cc_elaborate( _cc_out );
      _cc_out << "," << NL();
      if( interface_decl->get_kind() == IIR_TERMINAL_INTERFACE_DECLARATION ) {
	_cc_out<< "IN" ;
      }
      else {
	switch( interface_decl->_get_mode() ){
	case IIR_IN_MODE:
	  _cc_out << "IN";
	  break;
	case IIR_OUT_MODE:
	  _cc_out << "OUT";
	  break;
	case IIR_INOUT_MODE:
	  _cc_out << "INOUT";
	  break;
	case IIR_BUFFER_MODE:
	  _cc_out << "BUFFER";
	  break;
	case IIR_LINKAGE_MODE:
	  _cc_out << "LINKAGE";
	  break;
	default:
	  cerr << "Warning!!! UnknownMode" << endl;
	  _cc_out << "IN";
	  break;
	};
      }
      _cc_out << CS(");");
      association = successor(association);
    }
  }
}

int
IIRScram_AssociationList::_publish_cc_unconstrained_ports( published_file &_cc_out,
							   IIR_Boolean commaFlag ) {
  int numUnconstrainedPorts       = 0;
  IIR_AssociationElement *element = first();

  while (element != NULL) {
    if (element->get_formal() != NULL) {
      ASSERT ( element->get_formal()->get_subtype() != NULL );
      if (element->get_formal()->get_subtype()->_is_unconstrained_array_type() == TRUE) {
	// An unconstrained formal must be constrained by an actual..
	
	ASSERT ( element->_get_actual() != NULL );

	if (commaFlag == TRUE) {
	  _cc_out << "," << NL();
	}
	
	element->_get_actual()->_publish_cc_elaborate( _cc_out );

	numUnconstrainedPorts++;
	commaFlag = TRUE;
      }
    }

    element = successor(element);
  }

  return numUnconstrainedPorts;
}
    
void
IIRScram_AssociationList::_publish_cc_elaborate( published_file &_cc_out ){
  IIR_InterfaceDeclaration* port_clause;
  IIR_AssociationElement* actual_clause;
  IIR* formal;
  IIR_Declaration* interfacedecl;
  IIR_Declaration* actualdecl;

  SCRAM_CC_REF( _cc_out, "IIRScram_AssociationList::_publish_cc_elaborate" );
  
  port_clause = ((IIR_ComponentDeclaration*)_get_current_publish_node())->local_port_clause.first();
  actual_clause = first();
  formal = actual_clause->get_formal();
  if(formal != NULL) {
    //### assumes the parameter in order, also on by expression not open yet
    for(; actual_clause != NULL; ) {
      interfacedecl = (IIR_Declaration*)actual_clause->get_formal();
      //->get_identifier()->_get_declaration();
      //      switch(((IIR_InterfaceDeclaration*)interfacedecl)->get_mode()) {
      //      case IIR_IN_MODE:
      //      case IIR_INOUT_MODE:
      _cc_out << "Add(";

      ASSERT ( ((IIR_AssociationElementByExpression *) actual_clause)->get_actual()->_is_iir_declaration() == TRUE );
      
      actualdecl = (IIR_Declaration *) ((IIR_AssociationElementByExpression *) actual_clause)->get_actual();
      
      ASSERT ( actualdecl->_is_iir_declaration() == TRUE );
      
      _publish_cc_scoping_prefix( _cc_out.get_stream(),
				  actualdecl->_get_declarative_region(), this );
      
      _cc_out << _get_current_elab_name();
      ((IIR_AssociationElementByExpression*)actual_clause)->get_actual()->_publish_cc_elaborate( _cc_out );
      _cc_out << "," << NL();
      _cc_out << _get_current_publish_name();
      actual_clause->get_formal()->_publish_cc_elaborate( _cc_out );
      _cc_out << ");" << NL();

      SCRAM_CC_REF( _cc_out, "IIRScram_AssociationList::_publish_cc_elaborate" );

      if((((IIR_InterfaceDeclaration*)interfacedecl)->get_mode() 
	  == IIR_OUT_MODE) || 
	 (((IIR_InterfaceDeclaration*)interfacedecl)->get_mode() 
	  == IIR_INOUT_MODE)) {
	_cc_out << "addChild(";

	ASSERT ( ((IIR_AssociationElementByExpression *) actual_clause)->get_actual()->_is_iir_declaration() == TRUE );
	actualdecl = (IIR_Declaration *) ((IIR_AssociationElementByExpression *) actual_clause)->get_actual();
	ASSERT ( actualdecl->_is_iir_declaration() == TRUE );
	  
	_publish_cc_scoping_prefix( _cc_out.get_stream(),
				    actualdecl->_get_declarative_region(), this );
	_cc_out << _get_current_elab_name();
	((IIR_AssociationElementByExpression*)actual_clause)->get_actual()->_publish_cc_elaborate( _cc_out );
	_cc_out << "," << NL();
	_cc_out << _get_current_publish_name();
	actual_clause->get_formal()->_publish_cc_elaborate( _cc_out );
	_cc_out << ");" << NL();
      }
      //	break;
      //      }
      actual_clause = successor(actual_clause);
    }
  }
  else {//### currently assumes parameters are in order. needs work
    while (port_clause != NULL) {
      //      switch(port_clause->get_mode()) {
      //      case IIR_IN_MODE:
      ///      case IIR_INOUT_MODE:
      _cc_out << "Add(";
      _cc_out << _get_current_elab_name();

      ASSERT ( ((IIR_AssociationElementByExpression *) actual_clause)->get_actual()->_is_iir_declaration() == TRUE );
      actualdecl = (IIR_Declaration *) ((IIR_AssociationElementByExpression *) actual_clause)->get_actual();
      ASSERT ( actualdecl->_is_iir_declaration() == TRUE );

      _publish_cc_scoping_prefix( _cc_out.get_stream(),
				  actualdecl->_get_declarative_region(), this );
      
      ((IIR_AssociationElementByExpression*)actual_clause)->get_actual()->_publish_cc_elaborate( _cc_out );
      _cc_out << "," << NL();
      _cc_out << _get_current_publish_name();
      port_clause->_get_declarator()->_publish_cc_elaborate( _cc_out );
      _cc_out << "_info);" << NL();
      
      SCRAM_CC_REF( _cc_out, "IIRScram_AssociationList::_publish_cc_elaborate" );

      if((port_clause->get_mode() == IIR_OUT_MODE) || 
	 (port_clause->get_mode() == IIR_INOUT_MODE)) {	
	_cc_out << "addChild(";
	_cc_out << _get_current_elab_name();

	ASSERT ( ((IIR_AssociationElementByExpression *) actual_clause)->get_actual()->_is_iir_declaration() == TRUE );
	actualdecl = (IIR_Declaration *) ((IIR_AssociationElementByExpression *) actual_clause)->get_actual();
	ASSERT ( actualdecl->_is_iir_declaration() == TRUE );
	
	_publish_cc_scoping_prefix( _cc_out.get_stream(),
				    actualdecl->_get_declarative_region(),this );
	
	((IIR_AssociationElementByExpression*)actual_clause)->get_actual()->_publish_cc_elaborate( _cc_out );
	_cc_out << "," << NL();
	_cc_out << _get_current_publish_name();
	port_clause->_get_declarator()->_publish_cc_elaborate( _cc_out );
	_cc_out << "_info);" << NL();
      }
      //	break;
      //      }
      port_clause = ((IIR_ComponentDeclaration*)_get_current_publish_node())->local_port_clause.successor((IIR_SignalInterfaceDeclaration*)port_clause);
      actual_clause = successor(actual_clause);
    }
  }
}

void
IIRScram_AssociationList::_publish_cc_necessary_temporaries_for_TC( published_file &_cc_out,
								    IIR_InterfaceList *formals_list) {

  IIR_AssociationElement   *call_parameter = first();
  IIR_InterfaceDeclaration *procedure_parameter;

  SCRAM_CC_REF( _cc_out, "IIRScram_AssociationList::_publish_cc_necessary_temporaries_for_TC" );

  ASSERT (formals_list->get_kind() == IIR_INTERFACE_LIST);
  procedure_parameter = formals_list->first();
  
  while (call_parameter != NULL) {
    if (call_parameter->get_formal()->get_kind() == IIR_FUNCTION_CALL) {
      _cc_out << procedure_parameter->get_subtype()->_get_cc_type_name();
      _cc_out << " ";
      procedure_parameter->_get_declarator()->_publish_cc_lvalue( _cc_out.get_stream() );
      _cc_out << "(ObjectBase::VARIABLE";
      if ((procedure_parameter->get_mode() == IIR_IN_MODE) || (procedure_parameter->get_mode() == IIR_INOUT_MODE)) {
	_cc_out << "," << NL();
	_cc_out << "(const ";
	call_parameter->_get_actual()->_publish_cc_kernel_type( _cc_out.get_stream() );
	_cc_out << " &) ";
	call_parameter->_get_actual()->_publish_cc_lvalue( _cc_out );
      }
      _cc_out << ");";
      _cc_out.new_line();
    }
    
    call_parameter = successor(call_parameter);
    procedure_parameter = formals_list->successor(procedure_parameter);
  }
}

void
IIRScram_AssociationList::_publish_cc_with_temporaries_for_TC( published_file &_cc_out,
							       IIR_InterfaceList *formals_list) {
  
  IIR_AssociationElement   *node = first();
  IIR_InterfaceDeclaration *proc_parameter;

  SCRAM_CC_REF( _cc_out, "IIRScram_AssociationList::_publish_cc_with_temporaries_for_TC" );

  ASSERT (formals_list->get_kind() == IIR_INTERFACE_LIST);
  proc_parameter = formals_list->first();
  
  if (node != NULL) {
    if (node->get_formal()->get_kind() == IIR_FUNCTION_CALL) {
      proc_parameter->_get_declarator()->_publish_cc_lvalue( _cc_out.get_stream() );
    }
    else {
      node->_publish_cc_lvalue( _cc_out );
    }

    proc_parameter = formals_list->successor(proc_parameter);
    
    for (node = successor(node); node != NULL; node = successor(node)) {
      _cc_out << ",";
      _cc_out.new_line();
      if (node->get_formal()->get_kind() == IIR_FUNCTION_CALL) {
	proc_parameter->_get_declarator()->_publish_cc_lvalue( _cc_out.get_stream() );
      }
      else {
	node->_publish_cc_lvalue( _cc_out );
      }
      proc_parameter = formals_list->successor(proc_parameter);
    }
  }
}

void
IIRScram_AssociationList::_publish_cc_restore_from_temporaries( published_file &_cc_out,
								IIR_InterfaceList * ){

  IIR_AssociationElement *call_parameter = first();
  const string _old_publish_prefix_string = _get_publish_prefix_string();

  SCRAM_CC_REF( _cc_out, "IIRScram_AssociationList::_publish_cc_restore_from_temporaries" );
  
  while (call_parameter != NULL) {
    if (call_parameter->get_formal()->get_kind() == IIR_FUNCTION_CALL) {
      _cc_out << "assignVariable(";
      call_parameter->_get_actual()->_publish_cc_first_objectParameter( _cc_out );
      _cc_out << ",";
      _cc_out.new_line();

      // This is done to supress publishing of prefix string as the
      // parameters are actually local and not is state
      _set_publish_prefix_string( "" );
      call_parameter->get_formal()->_publish_cc_lvalue( _cc_out );
      _set_publish_prefix_string( _old_publish_prefix_string );
      
      _cc_out << ", nullInfo, nullInfo)";
      _cc_out.end_statement();
    }
    
    call_parameter = successor(call_parameter);
  }
}

void
IIRScram_AssociationList::_publish_cc_array_index( published_file &_cc_out ){
  SCRAM_CC_REF( _cc_out, "IIRScram_AssociationList::_publish_cc_array_index" );
  _publish_cc_elaborate( _cc_out );
}

void
IIRScram_AssociationList::_set_passed_through_out_port(IIR_Boolean) {
  // The parameter is just to keep the interface consistent

  IIR_AssociationElement *temp_association = first();
  IIR_AssociationElementByExpression *portAssociation;
  
  while (temp_association != NULL) {
    if( temp_association->get_kind() == IIR_ASSOCIATION_ELEMENT_BY_EXPRESSION ){
      portAssociation = (IIR_AssociationElementByExpression *) temp_association;
    
      switch (portAssociation->get_formal()->_get_mode()) {
      case IIR_IN_MODE:
	break;
      
      case IIR_OUT_MODE:
	if ((portAssociation->_get_actual() != NULL) &&
	    (portAssociation->_get_actual()->_is_signal() == TRUE)) {
	  portAssociation->_get_actual()->_set_passed_through_out_port(TRUE);
	}
	else {
	  cerr << "Warning : "
	       << "IIRScram_AssociationList::_set_passed_through_out_port()"
	       << " - Unhandled case in IIR_OUT_MODE" << endl;
	}
	break;
      
      case IIR_INOUT_MODE:
	if (portAssociation->_get_actual() == NULL) {
	  break;
	}
      
	if (portAssociation->_get_actual()->get_kind() ==
	    IIR_SIGNAL_DECLARATION) {
	  ((IIR_SignalDeclaration *) portAssociation->_get_actual())->
	    _set_passed_through_out_port(TRUE);
	}
	else if(portAssociation->_get_actual()->get_kind() == IIR_FUNCTION_CALL){
	  IIR_FunctionCall *tempFC = (IIR_FunctionCall *) portAssociation->_get_actual();
	  if (tempFC->parameter_association_list.first()->get_kind() ==
	      IIR_SIGNAL_DECLARATION) {
	    ((IIR_SignalDeclaration *) tempFC->parameter_association_list.first())->_set_passed_through_out_port(TRUE);
	  }
	}
	else if (portAssociation->_get_actual()->get_kind() ==
		 IIR_SIGNAL_INTERFACE_DECLARATION) {
	  // Nothing to do here. Check just to suppress the warning from
	  // popping up.
	}
	else if ((portAssociation->_get_actual()->get_kind() == IIR_INDEXED_NAME)
		 || (portAssociation->_get_actual()->get_kind()
		     == IIR_SELECTED_NAME)) {
	  portAssociation->_get_actual()->_set_passed_through_out_port(TRUE);
	}
	else {
	  cerr << "Warning : "
	       << "IIRScram_AssociationList::_set_passed_through_out_port()"
	       << " - Unhandled case in IIR_INOUT_MODE" << endl;
	}
	break;
      default:
	// Do nothing.
	break;
      }
    }
    else{
      string error = string("Unhandled association type ") + temp_association->get_kind_text() +
	" in IIRScram_AssociationList::_set_passed_through_out_port";
      report_error( temp_association, error );
      abort();
    }
    temp_association = successor(temp_association);
  }
}

ostream &
IIRScram_AssociationList::_print( ostream &os ){
  
  IIR_AssociationElement *current = first();
  if( current != NULL ){
    os << *current;
  }
  while( current != NULL ){
    current = successor( current );
    if( current != NULL ){
      os << ", " << *current;
    }
  }
  
  return os;
}

IIR_Boolean
IIRScram_AssociationList::_is_resolved(){
  IIR_Boolean retval = TRUE;

  IIR_AssociationElement *current = first();
  while( current != NULL ){
    if( current->_is_resolved() == FALSE ){
      retval = FALSE;
    }

    current = successor( current );
  }
  
  return retval;
}

void 
IIRScram_AssociationList::_add_decl_into_cgen_symbol_table() {
  IIR_AssociationElement *current = first();
  while( current != NULL ){
    if(current->get_kind() == IIR_ASSOCIATION_ELEMENT_BY_EXPRESSION) {
      current->_add_decl_into_cgen_symbol_table();
    }
    current = successor( current );
  }
}

void
IIRScram_AssociationList::_build_sensitivity_list(IIR_DesignatorList* sensitivity_list) {
  IIR_AssociationElement *current = first();
  while( current != NULL ) {
    current->_build_sensitivity_list(sensitivity_list);
    current = successor( current );
  }
}

void
IIRScram_AssociationList::
_build_above_attribute_set(set<IIR_AboveAttribute> *to_build) {
  IIR_AssociationElement *node = first();
  for (;node != NULL; node = successor(node)) {
    node->_build_above_attribute_set(to_build);
  }
}

IIR_Int32
IIRScram_AssociationList::_process_positional_part( IIR_InterfaceList *formal_list, 
						    IIR_Boolean resolve ){
  IIR_Int32 num_resolved = 0;
  // This method will return the position number of the last positional element processed.
  int local_count = 0;
  IIR_InterfaceDeclaration *current_formal = formal_list->first();
  IIR_AssociationElement *current_local = first();
  while( current_local != NULL && 
	 current_local->_is_positional() == TRUE ){

    IIR_Boolean found_match = FALSE;
    IIR_TypeDefinition *current_local_type = NULL;

    if( current_local->get_kind() == IIR_ASSOCIATION_ELEMENT_BY_EXPRESSION ){
      // Process all of the positional associations
      if( current_formal != NULL && found_match == FALSE ){
	// Does the type of our local match that of the parameter?
	IIR_TypeDefinition *current_formal_type = current_formal->get_subtype();
	set<IIR_TypeDefinition> *current_local_rvals = current_local->_get_rval_set();
	if( current_local_rvals == NULL ){
	  if( resolve == TRUE ){
	    ostringstream err;
	    err << "Internal error in IIRScram_AssociationList::_process_positional_part"
		<< "- no rval found for an local!";
	    report_error( current_local, err.str() );
	    abort();
	  }
	  else{
	    return -1;
	  }
	}
	current_local_type = current_local_rvals->get_element();
	while( current_local_type != NULL ){
	  if( current_local_type->_is_compatible( current_formal_type ) != NULL ){
	    found_match = TRUE;
	    break;
	  }
	  current_local_type = current_local_rvals->get_next_element();
	}
	
	delete current_local_rvals;

	if( found_match == TRUE ){
	  if( resolve == TRUE ){
	    current_local->set_formal( current_formal );      
	    IIR *new_actual;
	    new_actual = current_local->_get_actual()->_semantic_transform( current_local_type );
	    new_actual->_type_check( current_local_type );
	    current_local->_set_actual( new_actual->_rval_to_decl( current_local_type ) );
	  }
	  num_resolved++;
	}
	else{
	  if( current_formal->_is_optional() == FALSE ){
	    if( resolve == TRUE ){
	      // Then something went wrong - we shouldn't have been called seeing
	      // how this interface list doesn't even go with this association list!
	      ostringstream err;
	      err << "Internal error in IIRScram_AssociationList::_process_positional_part"
		  << "- no matched found for required local!";
	      report_error( current_local, err.str() );
	      abort();
	    }
	    else{
	      // we're not resolving - we're just checking.
	      return -1;
	    }
	  }
	}
      }
    }
    else{
      ASSERT( current_local->get_kind() == IIR_ASSOCIATION_ELEMENT_BY_OPEN);
      num_resolved++;
      if( resolve == TRUE ){
	current_local->set_formal( current_formal );            
      }
    }
  
    current_local = successor( current_local );
    // Either we got a match or the parameter is optional, or we would have aborted.
    local_count++;
    if( current_formal != NULL ){
      current_formal = formal_list->successor(current_formal);
    }
    else{
      break;
    }
  }

  return num_resolved;
}

void 
IIRScram_AssociationList::_resolve_and_order( IIR_InterfaceList *formal_list, 
					      IIR_InterfaceList *local_list,
					      IIR *line_info ){
  IIR_Boolean result = _check_or_resolve( formal_list, local_list, TRUE );
  if( result == TRUE ){
    _fill_in_defaults( line_info, formal_list );
    ASSERT( _is_resolved() == TRUE );
  }
}

IIR_Boolean 
IIRScram_AssociationList::_check_valid_arguments( IIR_InterfaceList *formal_list,
						  IIR_InterfaceList *local_list ){
  //  ASSERT( _is_resolved() == FALSE );
  return _check_or_resolve( formal_list, local_list, FALSE );
}

IIR_Name *
IIRScram_AssociationList::_get_formal_designator_from_indexed_name( IIR_IndexedName *name ){

  IIR_Name *retval = name;
  // An indexed name in the formal can be many things.  It can be:
  // (pg. 62 of the LRM)
  // conversion_function( formal_designator )
  // type_conversion( formal_designator )
  // or 
  // formal_designator
  // The formal designator can an array access or a slice, too.

  // Let's look up the prefix, and if we find a type or function
  // declaration for the prefix, then the suffix is the formal designator.
  // Otherwise, the whole indexed name is.


  IIR_Name *to_lookup = name;
  while( to_lookup->get_kind() == IIR_INDEXED_NAME ){
    ASSERT( to_lookup->get_prefix() != NULL );
    ASSERT( to_lookup->get_prefix()->_is_iir_name() == TRUE );
    to_lookup = (IIR_Name *)to_lookup->get_prefix();
  }
  
  IIR_Boolean suffix_is_formal_designator = FALSE;
  set<IIR_Declaration> *prefix_decls = to_lookup->_symbol_lookup();
  if( prefix_decls == NULL ){
    // Then we couldn't find the prefix in the symbol table.  It's not 
    // a conversion of any sort.
    return retval;
  }

  IIR_Declaration *current_decl = prefix_decls->get_element();
  while( current_decl != NULL ){
    if( (current_decl->_is_iir_function_declaration() == TRUE && 
	 ((IIR_FunctionDeclaration *)current_decl)->_could_be_conversion_function() == TRUE) ||
	current_decl->_is_type() == TRUE ){
      // Then the suffix is the formal designator
      suffix_is_formal_designator = TRUE;
      break;
    }
    current_decl = prefix_decls->get_next_element();
  }
  
  delete prefix_decls;

  if( suffix_is_formal_designator == TRUE ){
    // If the suffix isn't a name, we're in trouble!
    IIR *temp = name->get_suffix();  
    if( temp->_is_iir_name() == FALSE ){
      retval = NULL;
    }
    else{
      retval = (IIR_Name *)temp;
    }
  }

  return retval;
}

IIR *
IIRScram_AssociationList::_get_actual_designator_from_indexed_name( IIR_IndexedName *name ){
  IIR *retval = name;
  IIR *prefix = name->get_prefix();
  IIR *suffix = name->get_suffix();

  if( suffix->get_kind() != IIR_ASSOCIATION_LIST ||
      ((IIR_AssociationList *)suffix)->num_elements() != 1 ){
    set<IIR_Declaration> *decls = prefix->_symbol_lookup();
    // might be type conversion or conversion function
    IIR_Boolean prefix_match = FALSE;
    if( decls != NULL ){
      IIR_Declaration *current = decls->get_element();
      while( current != NULL ){
	if( current->get_kind() == IIR_FUNCTION_DECLARATION &&
	    ((IIR_FunctionDeclaration *)current)->_could_be_conversion_function() == TRUE ){
	  prefix_match = TRUE;
	  break;
	}
	else if( current->get_kind() == IIR_TYPE_DECLARATION ||
		 current->get_kind() == IIR_SUBTYPE_DECLARATION ){
	  prefix_match = TRUE;
	  break;
	}
	current = decls->get_next_element();
      }
    }
    
    if( prefix_match == TRUE ){
      retval = suffix;
    }
    delete decls;
  }

  return retval;
}

IIR_Boolean
IIRScram_AssociationList::_check_or_resolve( IIR_InterfaceList *formal_list,
					     IIR_InterfaceList *local_list,
					     IIR_Boolean resolve ){
  
  int num_resolved = _process_positional_part( formal_list, resolve );
  if( num_resolved == -1 ){
    // This means there was a problem with the VHDL.
    if( resolve == TRUE ){
      ostringstream err;
      err << "Internal error in IIRScram_AssociationList::_check_or_resolve -"
	  << " _process_positional_part returned -1 when resolving.";
      report_error( this, err.str() );
      abort();
    }
    else{
      return FALSE;
    }
  }
  // If we made it here, the positional part of the VHDL was just fine.
  // Now lets check to see if we've got any more positional associations
  // after this.  We're not allowed to.
  IIR_AssociationElement *first_to_look_at = get_nth_element( num_resolved );
  IIR_AssociationElement *current_association = first_to_look_at;
  while( current_association != NULL ){
    if( current_association->_is_positional() == TRUE ){
      if( resolve == TRUE ){
	ostringstream err;
	err << "Positional associations cannot follow named associations"
	    << " in an association list.";
	report_error( current_association, err.str() );
      }
      else{
	return FALSE;
      }
    }
    current_association = successor( current_association );
  }

  // Now, we need to start looking at the named associations, and picking
  // out the formals.  The formals may be a simple name from an interface
  // list, a conversion function, or, a type conversion.
  current_association = first_to_look_at;
  while( current_association != NULL){
    IIR *formal_designator = NULL;
    IIR *actual_designator = NULL;
    IIR *formal_part = NULL;
    IIR *actual_part = NULL;
    // These are declarations that were found through selection, and
    // MADE VISIBLE.  This means that if they are non-NULL, that we
    // need to make them not visible at the end.
    IIR_Declaration *actual_declaration = NULL;
    IIR_Declaration *formal_declaration = NULL;

    // (We already looked for NULL formals, and if we made it here, there
    // shouldn't be any.
    formal_part = current_association->get_formal();
    ASSERT( formal_part != NULL );
    // So, we have a formal.  But, the formal MIGHT have a conversion
    // function associated with it.  So, if we have an indexed name,
    // we must look at the suffix to determine if the formal is there.
    
    formal_designator = _find_formal_designator( formal_part );
    if( formal_designator == NULL ){
      if( resolve == TRUE ){
	ostringstream err;
	err << "Couldn't find formal designator in IIRScram_AssociationList::_check_or_resolve"
	    << ", and resolve is TRUE!";
	report_error( this, err.str() );
	return FALSE;
      }
      else{
	return FALSE;
      }
    }
    
    formal_declaration = _find_formal_declaration( formal_designator, formal_list );
    if( formal_declaration == NULL ){
      if( resolve == TRUE ){
	ostringstream err;
	err << "|" << *formal_designator << "| does not designate a formal parameter in "
	    << "this scope.";

	report_error( this, err.str() );
	return FALSE;
      }
      else{
	return FALSE;
      }
    }

    ASSERT( formal_declaration->_is_iir_interface_declaration() == TRUE || 
	    formal_declaration->_is_iir_element_declaration() == TRUE  );

    // At this point, we have the formal declaration in
    // "formal_declaration", and we have the formal part in "formal_part".
    // We still want to determine the type of the formal part - this is
    // something that's hard to do - we can't just call "_get_rval_set" on
    // the formal part, because:
    // 1) If the formal is visible only through selection, it will fail to
    //    find the type.
    // 2) If there are conversion functions on both sides, we want the type
    //    of the declaration.  Otherwise, we want the type of the whole
    //    formal part....
    
    actual_part = current_association->_get_actual();
    IIR_Boolean need_to_hide_formal = FALSE;
    IIR_Boolean need_to_hide_actual = FALSE;

    if( actual_part != NULL ){
      // Then it's not open.
      actual_designator = _find_actual_designator( actual_part );
      
      if( local_list != NULL && actual_part->get_kind() == IIR_SIMPLE_NAME ){
	// There are only certain places that an actual by selection can
	// appear.  Looking at page 62 of the 93 LRM, do the substitution
	// of "signal_name" for "actual_designator" in the grammar, and
	// that's where it could be.  So, it could be in a name, (like the
	// prefix of an indexed name), or simple name, the suffix of an
	// indexed name (as part of a type conversion or conversion
	// function.)  So, it must be part of a name.
	
	if( actual_designator != NULL && actual_designator->_is_iir_name() == TRUE ){
	  // Now, let's see if we can find it in the interface list.
	  actual_declaration = _find_declaration( (IIR_Name *)actual_designator,
						  local_list );
	}
      }      
    }

    set<IIR_TypeDefinition> *formal_types_to_consider = new set<IIR_TypeDefinition>;
    need_to_hide_formal = _find_formal_types( current_association,
					      actual_part, 
					      actual_designator,
					      formal_part,
					      formal_designator,
					      formal_declaration,
					      formal_types_to_consider );
    
    if( formal_types_to_consider == NULL ){
      if( resolve == TRUE ){
	report_undefined_symbol( formal_part );
	return FALSE;
      }
      else{
	return FALSE;
      }
    }
    
    if( actual_part != NULL ){
      // Now we can fully resolve the actual, using the information we have
      // about the formal from before.
    
      if( actual_declaration != NULL ){
	// We're going to put this SINGLE declaration back into visibility
	// so that the type checking that occurs after this can see it.
	if( current_association->_get_symbol_table()->is_visible( actual_declaration ) == FALSE ){
	  current_association->_get_symbol_table()->make_visible( actual_declaration );
	  need_to_hide_actual = TRUE;
	}
      }
    
      set<IIR_TypeDefinition> *actual_part_rvals = actual_part->_get_rval_set();
      if( actual_part_rvals == NULL ){
	if( resolve == TRUE ){
	  report_undefined_symbol( actual_part );
	}
	return FALSE;
      }

      reconcile_sets( formal_types_to_consider, actual_part_rvals );
      ASSERT( formal_types_to_consider->num_elements() <= 1 );
      switch( formal_types_to_consider->num_elements() ){
      case 0:{
	if( resolve == TRUE ){
	  ostringstream err;
	  err << "Type of formal designator |" << *formal_designator << "| and actual part |"
	      << *actual_part << "| aren't compatible, as they must in this context.";
	  report_error( formal_designator, err.str() );
	}
	return FALSE;
	break;
      }
      case 1:{
	IIR_TypeDefinition *actual_part_type = formal_types_to_consider->get_element();
	// Just trying to be efficient with set constructors, here.
	IIR *new_actual = 0;

	// We have to do this this way, because "actual_part" needs to
	// be resolved no matter what.  But, we don't want it resolved
	// if we're not resolving, so we don't set it in the association
	// unless we're supposed to.  Also, "semantic transform" deletes
	// the old value in some cases, so we need to clone it if we
	// want to save it.
	if( resolve == FALSE ){
	  new_actual = actual_part->_clone();
	}

	actual_part = actual_part->_semantic_transform( formal_types_to_consider );
	actual_part->_type_check( formal_types_to_consider );
	actual_part = actual_part->_rval_to_decl( actual_part_type );

	if( resolve == TRUE ){
	  ASSERT( actual_part->_is_resolved() == TRUE );
	  
	  new_actual = actual_part;
	}
	current_association->_set_actual( new_actual );
      }
      }

      if( need_to_hide_actual == TRUE ){
	//	current_association->_get_symbol_table()->remove_from_visibility( actual_declaration );
	current_association->_get_symbol_table()->remove_from_scope( actual_declaration );
      }

      // Now we can fully resolve the formal, given the data in "actual_designator".
      set<IIR_TypeDefinition> *actual_types_to_consider;
      if( (formal_part != formal_designator) && (actual_part != actual_designator) ){
	actual_types_to_consider = actual_designator->_get_rval_set();
	ASSERT( actual_types_to_consider != NULL );
      }
      else{
	ASSERT( actual_part->_is_resolved() == TRUE );
	actual_types_to_consider = new set<IIR_TypeDefinition>( actual_part->get_subtype() );
      }

      if( current_association->_get_symbol_table()->is_visible( formal_declaration ) == FALSE ){
	current_association->_get_symbol_table()->make_visible( formal_declaration );
	need_to_hide_formal = TRUE;
      }

      set<IIR_TypeDefinition> *formal_part_rvals = formal_part->_get_rval_set();

      if( formal_part_rvals == NULL ){
	if( resolve == TRUE ){
	  report_undefined_symbol( formal_part );
	}
	return FALSE;
      }

      reconcile_sets( actual_types_to_consider, formal_part_rvals );
      ASSERT( actual_types_to_consider->num_elements() <= 1 );
      switch( actual_types_to_consider->num_elements() ){
      case 0:{
	if( resolve == TRUE ){
	  ostringstream err;
	  err << "Type of formal part |" << *formal_designator << "| and actual designator |"
	      << *actual_designator << "| don't match, as they must in this context.";
	  report_error( this, err.str() );
	}
	else{
	  return FALSE;
	}
	break;
      }
      case 1:{
	if( resolve == TRUE ){
	  formal_part = formal_part->_semantic_transform( actual_types_to_consider );
	  formal_part->_type_check( actual_types_to_consider );
	  formal_part = formal_part->_rval_to_decl( actual_types_to_consider->get_element() );      
	  ASSERT( formal_part->_is_resolved() == TRUE );
	  current_association->set_formal( formal_part );
	}
      }
	break;
      }

      if( resolve == TRUE ){
	ASSERT( current_association->_is_resolved() == TRUE );
      }

      delete formal_part_rvals;
      delete actual_part_rvals;
      delete actual_types_to_consider;
    }
    else{
      // This means this assocation is open We need to check and see if
      // this interface is allowed to be open...
      ASSERT( formal_declaration != NULL );

      if( current_association->_get_symbol_table()->is_visible( formal_declaration ) == FALSE ){
	current_association->_get_symbol_table()->make_visible( formal_declaration );
	need_to_hide_formal = TRUE;
      }

      ASSERT( formal_part == formal_designator );
      switch( formal_types_to_consider->num_elements() ){
      case 0:{
	if( resolve == TRUE ){
	  ostringstream err;
	  err << "Internal error in IIR_AssociationList::_check_or_resolve - "
	      << "found no formal types for open association |" << *current_association
	      << "|.";
	  report_error( current_association, err.str() );
	  abort();
	}
	return FALSE;
      }
      case 1:{
	if( resolve == TRUE ){
	  formal_part = formal_part->_semantic_transform( formal_types_to_consider );
	  formal_part->_type_check( formal_types_to_consider );
	  formal_part = formal_part->_rval_to_decl( formal_types_to_consider->get_element() );      
	  ASSERT( formal_part->_is_resolved() == TRUE );
	  current_association->set_formal( formal_part );
	}
	// else everything is OK, go to the next association
	break;
      }
      default:{
	if( resolve == TRUE ){
	  report_ambiguous_error( formal_part, formal_types_to_consider );
	}
        delete formal_types_to_consider;
	return FALSE;
      }
      }
    }

    if( need_to_hide_formal == TRUE ){
      //      current_association->_get_symbol_table()->remove_from_visibility( formal_declaration );
      current_association->_get_symbol_table()->remove_from_scope( formal_declaration );
    }
    delete formal_types_to_consider;

    current_association = successor( current_association );
  }

#ifdef DEVELOPER_ASSERTIONS
  if( resolve == TRUE ){
    ASSERT( _is_resolved() == TRUE );
    IIR_AssociationElement *current = first();
    while( current != NULL ){
      ASSERT( current->get_formal() != NULL );
      // This isn't true if it's OPEN
      //      ASSERT( current->_get_actual() != NULL );
      //      ASSERT( current->get_kind() == IIR_ASSOCIATION_ELEMENT_BY_EXPRESSION );
      ASSERT( current->_is_resolved() == TRUE );
      // This is good for debugging, but it's not always true.  (For
      // example, it's not true for a recursive function call.)
      //      ASSERT( current->get_formal() != current->_get_actual() );
      current = successor( current );
    }
  }
#endif
  
  // OK, now we need to make sure that every required formal has an local
  // associated with it.  We already know that all locals supplied match
  // their parameters, or we wouldn't have made it down here.
  int num_required = 0;
  IIR_InterfaceDeclaration *current_formal = formal_list->first();
  while( current_formal != NULL ){
    if( current_formal->_is_optional() == FALSE ){
      num_required++;
    }
    current_formal = formal_list->successor( current_formal );
  }

  if( num_elements() < num_required ){
    if( resolve == FALSE ){
      return FALSE;
    }
  }

  return TRUE;
}

IIR_InterfaceDeclaration *
IIRScram_AssociationList::_find_declaration_in_formal( IIR_AssociationElement *find_in ){
  IIR_Declaration *retval = NULL;

  IIR *current_formal = find_in->get_formal();  
  ASSERT( current_formal != NULL );

  retval = current_formal->_find_formal_declaration();

  ASSERT( retval != NULL );
  ASSERT( retval->_is_iir_interface_declaration() == TRUE );
  
  return (IIR_InterfaceDeclaration *)retval;
}

void 
IIRScram_AssociationList::_fill_in_defaults( IIR *line_info, IIR_InterfaceList *formal_list ){
  ASSERT( formal_list != NULL );
  ASSERT( _is_resolved() == TRUE );

  set<IIR_InterfaceDeclaration> parameter_declarations;
  
  // Build a set of the parameter declarations.
  IIR_InterfaceDeclaration *current_formal = formal_list->first();
  while( current_formal != NULL ){
    parameter_declarations.add( current_formal );
    current_formal = formal_list->successor( current_formal );
  }
  
  // Walk the list of locals, and remove each parameter found from the
  // set...
  IIR_AssociationElement *current_local = first();
  while( current_local != NULL ){
    parameter_declarations.remove( _find_declaration_in_formal( current_local ) );
    current_local = successor( current_local );
  }

  // Now, we have all of the parameters with no association in the set 
  // "parameter_declarations".  We need to build an association for each, and
  // append it to the list.

  current_formal = parameter_declarations.get_element();
  while( current_formal != NULL ){
    IIR_AssociationElement *new_association = 0;
    if( current_formal->get_value() ){
      new_association = new IIR_AssociationElementByExpression();
      static_cast<IIR_AssociationElementByExpression *>(new_association)->set_actual( current_formal->get_value());
    }
    else{
      new_association = new IIR_AssociationElementOpen();
    }
    copy_location( line_info, new_association );
    new_association->set_formal( current_formal );

    append( new_association );

    current_formal = parameter_declarations.get_next_element();
  }

  IIR_AssociationList *new_list = new IIR_AssociationList();
  // copy_location( this, new_list );

  // Now we're going to reorder ourself into the order of the interface list.
  if( num_elements() == formal_list->num_elements() ){
    // The condition above won't hold in the case of associating indexed names, slices,
    // or record elements.
    IIR_InterfaceDeclaration *current_decl = formal_list->first();
    while( current_decl != NULL ){
      IIR_AssociationElement *current_assoc = first();
      IIR_Boolean one_matched = FALSE;
      while( current_assoc != NULL ){
	IIR_InterfaceDeclaration *formal_declaration = _find_declaration_in_formal( current_assoc );
	if( formal_declaration == current_decl ){
	  one_matched = TRUE;
	  break;
	}
	current_assoc = successor( current_assoc );      
      }
      if( one_matched == TRUE ){
	// current_assoc holds the match
	new_list->append( current_assoc );
	remove( current_assoc );
      }
      else{
	ostringstream err;
	err << "Internal error in IIRScram_AssociationList::_fill_in_defaults - trying"
	    << " to reorder list and can't find declaration |" << *current_decl << "|";
	report_error( this, err.str() );
      }
      current_decl = formal_list->successor( current_decl );
    }
    *this = *new_list;
    delete new_list;
    ASSERT( num_elements() == formal_list->num_elements() );
  }
}

IIR *
IIRScram_AssociationList::_clone(){
  IIR_AssociationList *retval = new IIR_AssociationList();
  IIRScram_List::_clone( retval );
  
  return retval;
}

IIR_Boolean  
IIRScram_AssociationList::_is_globally_static_for_array(){
  IIR_Boolean retval = TRUE;
  IIR_AssociationElement *current = first();
  while( current != NULL ){
    if( current->_is_globally_static_for_array() == FALSE ){
      retval = FALSE;
      break;
    }
    current = successor( current );
  }
  
  return retval;
}

IIR_Boolean  
IIRScram_AssociationList::_is_globally_static_for_record( ){
  IIR_Boolean retval = TRUE;
  IIR_AssociationElement *current = first();
  while( current != NULL ){
    if( current->_is_globally_static_for_record() == FALSE ){
      retval = FALSE;
      break;
    }
    current = successor( current );
  }
  
  return retval;
}
 
IIR_Boolean 
IIRScram_AssociationList::_is_globally_static_for_function( ){
  IIR_Boolean retval = TRUE;
  IIR_AssociationElement *current = first();
  while( current != NULL ){
    if( current->_is_globally_static_for_function() == FALSE ){
      retval = FALSE;
      break;
    }
    current = successor( current );
  }
  
  return retval;
}

IIR_Boolean
IIRScram_AssociationList::_is_static_expression() {
  return _is_globally_static_for_function();
}

IIR_Boolean
IIRScram_AssociationList::_are_all_scalars() {
  IIR_AssociationElement *current = first();
  while( current != NULL ) {
    ASSERT ( current->get_formal() != NULL );
    if( current->get_formal()->_is_scalar_type() == FALSE ){
      return FALSE;
    }
    current = successor( current );
  }
  
  return TRUE;
}

IIR_Boolean 
IIRScram_AssociationList::_is_above_attribute_found() {
  IIR_Boolean retval = FALSE;
  IIR_AssociationElement *node = first();
  for (;node != NULL; node = successor(node)) {
    retval = retval || node->_is_above_attribute_found();
  }
  return retval;
}

IIR_Declaration *
IIRScram_AssociationList::_find_declaration( IIR_Name *to_find, IIR_DeclarationList *list ){
  ASSERT( to_find != NULL );
  ASSERT( list != NULL );

  IIR_Declaration *retval = NULL;
    
  set<IIR_Declaration> *poss_decls = list->_find_declarations( to_find );
  if( poss_decls != NULL ){
    ASSERT( poss_decls->num_elements() == 1 );
    retval = poss_decls->get_element();
    delete poss_decls;
  }
  
  return retval;
}

IIR *
IIRScram_AssociationList::_find_formal_designator( IIR *formal_part ){
  ASSERT( formal_part != 0 );

  IIR *retval = NULL;
  if( formal_part->_is_resolved() ){
    retval = formal_part;
  }
  else{
    // According to Ashenden's book (pg. 594) the formal may only legally
    // be a generic name, a formal name, a paramter name, a function name,
    // or a type conversion.  This limits it to a simple name or an indexed
    // name.  (could be work.typename( formal_designator ), but this is
    // still an indexed name.
    switch( formal_part->get_kind() ){
    case IIR_SIMPLE_NAME:
    case IIR_SELECTED_NAME:
      retval = (IIR_Name *)formal_part;
      break;
    case IIR_INDEXED_NAME:
      // I need to pass the list, to be able to find it in the suffix OR prefix.
      retval = _get_formal_designator_from_indexed_name( (IIR_IndexedName *)formal_part );
      break;
      
    default:
      // The parser handed us illegal syntax.
      ostringstream err;
      err << "Syntax error at |" << *formal_part << "| - expecting formal parameter,"
	  << "generic, port, type conversion, or conversion function.";
      report_error( formal_part, err.str() );
    }    
  }
  return retval;
}

IIR_Declaration *
IIRScram_AssociationList::_find_formal_declaration( IIR *formal_designator,
						    IIR_InterfaceList *formal_list ){
  IIR_Declaration *retval = NULL;

  // Now we have the name of the formal in "formal_designator" - we need
  // to search the paramter for the type of the formal.  NOTE that if we
  // had "integer( x )", we'd have "x" in "formal_designator".  Same goes
  // for a conversion function.
  if( formal_designator->_is_iir_declaration() ){
    retval = static_cast<IIR_Declaration *>( formal_designator );
  }
  else if( formal_designator->get_kind() == IIR_INDEXED_NAME ){
    IIR *prefix = formal_designator;
    while( prefix->get_kind() == IIR_INDEXED_NAME ){
      ASSERT( dynamic_cast<IIR_IndexedName *>(prefix)->get_prefix() != NULL );
      prefix = dynamic_cast<IIR_IndexedName *>(prefix)->get_prefix();
    }
    
    // This method searches a declaration list for a SINGLE match.
    ASSERT( prefix->_is_iir_name() );
    retval = _find_declaration( dynamic_cast<IIR_Name *>(prefix), formal_list );
    if( retval == NULL ){
      ASSERT( formal_designator->_is_iir_name() );
      // The formal wasn't in the prefix - look in the suffix.
      IIR *suffix = dynamic_cast<IIR_Name *>(formal_designator)->_get_suffix();
      ASSERT( suffix->_is_iir_name() == TRUE );
      retval = _find_declaration( (IIR_Name *)suffix, formal_list );
    }
  }
  else{
    ASSERT( formal_designator->_is_iir_name() );    
    retval = _find_declaration( dynamic_cast<IIR_Name *>(formal_designator), formal_list );
  }

  return retval;
}

IIR *
IIRScram_AssociationList::_find_actual_designator( IIR *actual_part ){
  IIR *retval = NULL;

  if( actual_part->get_kind() == IIR_INDEXED_NAME ){
    retval = _get_actual_designator_from_indexed_name( (IIR_IndexedName *)actual_part );
  }
  else{
    retval = actual_part;
  }

  return retval;
}

bool 
IIRScram_AssociationList:: _find_formal_types( IIR_AssociationElement *current_association,
					       IIR *actual_part, 
					       IIR *actual_designator,
					       IIR *formal_part,
					       IIR *formal_designator,
					       IIR_Declaration *formal_declaration,
					       set<IIR_TypeDefinition> *add_types_to_me ){

  bool retval = FALSE;

  ASSERT( formal_part != NULL );
  ASSERT( formal_declaration != NULL );
  ASSERT( add_types_to_me != NULL && add_types_to_me->num_elements() == 0 );
  
  if( (formal_designator != formal_part) && (actual_designator != actual_part ) ){
    add_types_to_me->add(formal_declaration->get_subtype());
  }
  else{
    if( current_association->_get_symbol_table()->is_visible( formal_declaration ) == FALSE ){
      current_association->_get_symbol_table()->make_visible( formal_declaration );
      retval = TRUE;
    }
    set<IIR_TypeDefinition> *formal_rvals = formal_part->_get_rval_set();
    if( formal_rvals != NULL ){
      add_types_to_me->add( formal_rvals );
      delete formal_rvals;
    }
    else{
      report_undefined_symbol( formal_part );
    }
  }

  if( retval == TRUE ){
    //    current_association->_get_symbol_table()->remove_from_visibility( formal_declaration );
    current_association->_get_symbol_table()->remove_from_scope( formal_declaration );
  }
  
  return retval;
}

void
IIRScram_AssociationList::_build_default_map( IIR_ComponentInstantiationStatement *err_info,
					      IIR_InterfaceList *formal_list,
					      IIR_InterfaceList *actual_list ){
  IIR_InterfaceDeclaration *current_actual = actual_list->first();
  while( current_actual != 0 ){
    if( !formal_list->_find_declaration( current_actual->get_declarator() ) ){
      string err;
      err = "No formal corresponds to actual |" + 
	current_actual->get_declarator()->_convert_to_string() + "|";
      report_error( err_info, err );
      return;
    }
    current_actual = actual_list->successor( current_actual );
  }

  IIR_InterfaceDeclaration *current_formal = formal_list->first();
  while( current_formal != 0 ){
    IIR_InterfaceDeclaration *actual_decl = 
      actual_list->_find_declaration( current_formal->get_declarator() );
    IIR_AssociationElement *new_association = 0;
    if( actual_decl ){
      new_association = new IIR_AssociationElementByExpression();
      new_association->set_formal( current_formal );
      new_association->_set_actual( actual_decl );
    }
    else{
      new_association = new IIR_AssociationElementOpen();
      new_association->set_formal( current_formal );
    }
    copy_location( err_info, new_association );
    append( new_association );

    current_formal = formal_list->successor( current_formal );
  }
  
}

visitor_return_type *
IIRScram_AssociationList::_accept_visitor( node_visitor *visitor, 
					   visitor_argument_type *arg ){
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_AssociationList(this, arg);
}
