#include <string.h>
#include <time.h>

#include <glib-object.h>
#include <glib.h>

#include "../kptraininglog.h"
#include "../kipina-i18n.h"
#include "../kputil.h"

#include "kpcalendarentryinfodialog.h"
#include "kpnewsplitworkoutdialog.h"
#include "kpnewworkoutdialog.h"
#include "kpnewcommentdialog.h"
#include "kpguiutils.h"
#include "kplistview.h"
#include "kpviewmodel.h"

/* Callbacks */
static void       kp_list_view_class_init       (KPListViewClass *klass);
static void       kp_list_view_init             (KPListView *lv);
static void       kp_list_view_model_init       (KPViewModelIface *iface);
static void       kp_list_view_finalize         (GObject *object);

/* Callback functions */
static void       view_popup_menu               (GtkWidget *treeview,
                                                 GdkEventButton *event,
                                                 gpointer data);
static void       on_popup_delete_clicked       (GtkWidget *menuitem,
                                                 KPListView *lv);
static void       on_popup_properties_clicked   (GtkWidget *menuitem,
                                                 KPListView *lv);
static gboolean   on_popup_menu                 (GtkWidget *treeview,
                                                 gpointer data);
static gboolean   on_button_press_event         (GtkWidget *treeview,
                                                 GdkEventButton *event,
                                                 gpointer data);
/* Follow log changes.. */
static void       log_connect_signals           (KPTrainingLog *log,
                                                 KPListView *lv);
static void       log_disconnect_signals        (KPTrainingLog *log,
                                                 KPListView *lv);
static void       log_changed                   (KPTrainingLog *log,
                                                 KPListView *lv);
static GtkWidget *create_popup_menu             (KPListView *lv);
static void       kp_list_view_set_dmy          (KPViewModel *view,
                                                 guint d,
                                                 guint m,
                                                 guint y);
static void       kp_list_view_get_dmy          (KPViewModel *view,
                                                 guint *d,
                                                 guint *m,
                                                 guint *y);
static void       kp_list_view_set_log          (KPViewModel *view,
                                                 KPTrainingLog *log);
static void       kp_list_view_unset_log        (KPViewModel *view);
static void       kp_list_view_set_view_type    (KPViewModel *view,
                                                 KPViewModelType type);
static
KPViewModelType   kp_list_view_get_view_type    (KPViewModel *view);
static void       kp_list_view_update_dates     (KPListView *view);
static void       kp_list_view_update_entries   (KPListView *lv);
static void       kp_list_view_update           (KPListView *lv);


typedef struct KPListViewPrivateData_
{
  GtkWidget        *popup_menu;

  GtkWidget        *popup_mi_properties;
  GtkWidget        *popup_mi_delete;
 
  GList            *entries;
  GtkListStore     *store;

  KPViewModelType   type;
  KPTrainingLog    *log;

  GDate            *date_s;
  GDate            *date_e;
  GDate            *date;
} KPListViewPrivateData;

#define KP_LIST_VIEW_PRIVATE_DATA(widget) (((KPListViewPrivateData*) \
      (KP_LIST_VIEW (widget)->private_data)))

enum {
  COLUMN_DATE,
  COLUMN_SPORT,
  COLUMN_DISTANCE,
  COLUMN_TIME,
  COLUMN_PACE,
  COLUMN_COMMENT,
  COLUMN_OTHER,
  COLUMN_TYPE,
  COLUMN_DESCRIPTION,
  COLUMN_N
};


GType
kp_list_view_get_type (void)
{
  static GType        kp_list_view_type = 0;

  if (!kp_list_view_type) {
    static const GTypeInfo kp_list_view_info = {
      sizeof (KPListViewClass),
      NULL,
      NULL,
      (GClassInitFunc) kp_list_view_class_init,
      NULL,
      NULL,
      sizeof (KPListView),
      0,
      (GInstanceInitFunc) kp_list_view_init,
      NULL
    };
    static const GInterfaceInfo view_model_info = {
      (GInterfaceInitFunc) kp_list_view_model_init,
      NULL,
      NULL
    };
    kp_list_view_type = g_type_register_static (GTK_TYPE_TREE_VIEW,
                                               "KPListView",
                                               &kp_list_view_info, 0);
    g_type_add_interface_static (kp_list_view_type,
                                 KP_TYPE_VIEW_MODEL,
                                &view_model_info);

  }
  return kp_list_view_type;
}

static void
kp_list_view_class_init (KPListViewClass *klass)
{
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = kp_list_view_finalize;
}

static void
distance_cell_func (GtkTreeViewColumn *column, GtkCellRenderer *renderer,
                    GtkTreeModel *model, GtkTreeIter *iter, gpointer data)
{
  gdouble distance;
  gchar buf[16];

  gtk_tree_model_get (model, iter, COLUMN_DISTANCE, &distance, -1);

  if (distance == (guint) distance)
    g_snprintf (buf, sizeof (buf), "%.0f", distance);
  else
    g_snprintf (buf, sizeof (buf), "%.3f", distance);

  g_object_set (renderer, "text", (distance) ? buf : "", NULL);
}

  
void
new_column (KPListView *lv, const gchar *title, guint column,
            GtkTreeCellDataFunc func)
{
  GtkTreeViewColumn *col;
  GtkCellRenderer *ren;

  ren = gtk_cell_renderer_text_new ();
  col = gtk_tree_view_column_new_with_attributes (title, ren, "markup", column,
                                                  NULL);
  gtk_tree_view_column_set_sort_column_id (col, column);
  
  gtk_tree_view_append_column (GTK_TREE_VIEW (lv), col);

  if (func)
    gtk_tree_view_column_set_cell_data_func (col, ren, func, NULL, NULL);

  if (column == COLUMN_DESCRIPTION)
    gtk_tree_view_column_set_visible (col, FALSE);

  gtk_tree_view_column_set_resizable (col, TRUE);
}


static void
kp_list_view_init (KPListView *lv)
{
  KPListViewPrivateData *p_data;
  GtkTreeSelection *selection;
  
  lv->private_data = g_new (KPListViewPrivateData, 1);
  p_data = KP_LIST_VIEW_PRIVATE_DATA (lv);
  p_data->date_s = g_date_new ();
  p_data->date_e = g_date_new ();
  p_data->date = g_date_new ();
  p_data->log = NULL;
  p_data->type = KP_VIEW_MODEL_TYPE_MONTH;
  p_data->entries = NULL;

  new_column (lv, _("Date"), COLUMN_DATE, NULL);
  new_column (lv, _("Sport"), COLUMN_SPORT, NULL);
  new_column (lv, _("Type"), COLUMN_TYPE, NULL);
  new_column (lv, _("Distance"), COLUMN_DISTANCE, distance_cell_func);
  new_column (lv, _("Time"), COLUMN_TIME, NULL);
  new_column (lv, _("Pace"), COLUMN_PACE, NULL);
  new_column (lv, _("Details"), COLUMN_OTHER, NULL);
  new_column (lv, _("Comment"), COLUMN_COMMENT, NULL);
  new_column (lv, _("Description"), COLUMN_DESCRIPTION, NULL);

  p_data->store = gtk_list_store_new (COLUMN_N,
                                      G_TYPE_STRING,
                                      G_TYPE_STRING,
                                      G_TYPE_DOUBLE,
                                      G_TYPE_STRING,
                                      G_TYPE_STRING,
                                      G_TYPE_STRING,
                                      G_TYPE_STRING,
                                      G_TYPE_STRING,
                                      G_TYPE_STRING);
  gtk_tree_view_set_model (GTK_TREE_VIEW (lv), GTK_TREE_MODEL (p_data->store));
  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (lv), TRUE);
  gtk_tree_view_set_headers_clickable (GTK_TREE_VIEW (lv), TRUE);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (lv));
  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
  
  g_signal_connect (G_OBJECT (lv), "button-press-event",
                    G_CALLBACK (on_button_press_event), NULL);
  g_signal_connect (G_OBJECT (lv), "popup-menu",
                    G_CALLBACK (on_popup_menu), NULL);

  p_data->popup_menu = create_popup_menu (lv);
}

static void
kp_list_view_model_init (KPViewModelIface *iface)
{
  iface->get_dmy = kp_list_view_get_dmy;
  iface->set_dmy = kp_list_view_set_dmy;
  iface->set_log = kp_list_view_set_log;
  iface->unset_log = kp_list_view_unset_log;
  iface->set_view_type = kp_list_view_set_view_type;
  iface->get_view_type = kp_list_view_get_view_type;
}

static void
kp_list_view_finalize (GObject *object)
{
  KPListViewPrivateData *p_data;

  p_data = KP_LIST_VIEW_PRIVATE_DATA (object);

  g_date_free (p_data->date);

  if (p_data->log) {
    log_disconnect_signals (p_data->log, KP_LIST_VIEW (object));
    p_data->log = NULL;
  }
  
  KP_LIST_VIEW (object)->private_data = NULL;
  g_free (KP_LIST_VIEW (object)->private_data);  
}


GtkWidget *
kp_list_view_new (void)
{
  KPListViewPrivateData *p_data;
  GtkWidget *widget;

  widget = g_object_new (kp_list_view_get_type (), NULL);

  p_data = KP_LIST_VIEW_PRIVATE_DATA (widget);
  
  gtk_widget_show_all (GTK_WIDGET (widget));
  
  return widget;
}

static void
kp_list_view_set_dmy (KPViewModel *view, guint d, guint m, guint y)
{
  KPListViewPrivateData *p_data;
  
  p_data = KP_LIST_VIEW_PRIVATE_DATA (view);

  g_return_if_fail (g_date_valid_dmy (d, m, y));
  g_date_set_dmy (p_data->date, d, m, y);

  kp_list_view_update (KP_LIST_VIEW (view));
  
  kp_debug ("Date set: %u.%u.%u", d, m, y);
}

static void
kp_list_view_get_dmy (KPViewModel *view, guint *d, guint *m, guint *y)
{
  KPListViewPrivateData *p_data;
  p_data = KP_LIST_VIEW_PRIVATE_DATA (view);

  g_return_if_fail (KP_IS_LIST_VIEW (view));
  g_return_if_fail (g_date_valid (p_data->date));
  
  if (d)
    *d = g_date_get_day (p_data->date);
  if (m)
    *m = g_date_get_month (p_data->date);
  if (y)
    *y = g_date_get_year (p_data->date);
}

static void
kp_list_view_set_view_type (KPViewModel *view, KPViewModelType type)
{
  KPListViewPrivateData *p_data;
  p_data = KP_LIST_VIEW_PRIVATE_DATA (view);
  p_data->type = type;

  if (type == KP_VIEW_MODEL_TYPE_ALL_TIME)
    kp_debug ("Set all the time on!");
  
  kp_list_view_update (KP_LIST_VIEW (view));
}


static KPViewModelType
kp_list_view_get_view_type (KPViewModel *view)
{
  KPListViewPrivateData *p_data;
  p_data = KP_LIST_VIEW_PRIVATE_DATA (view);
  
  return p_data->type;
}


static void
kp_list_view_set_log (KPViewModel *model, KPTrainingLog *log)
{
  KPListViewPrivateData *p_data;

  g_return_if_fail (KP_IS_TRAINING_LOG (log));
  
  p_data = KP_LIST_VIEW_PRIVATE_DATA (model);
  p_data->log = log;

  kp_list_view_update (KP_LIST_VIEW (model));

  log_connect_signals (log, KP_LIST_VIEW (model));
}

static void
kp_list_view_unset_log (KPViewModel *model)
{
  KPListViewPrivateData *p_data;
  p_data = KP_LIST_VIEW_PRIVATE_DATA (model);

  gtk_list_store_clear (p_data->store);
  
  log_disconnect_signals (p_data->log, KP_LIST_VIEW (model));
  p_data->log = NULL;
}

static void
kp_list_view_update_dates (KPListView *view)
{
  KPListViewPrivateData *p_data;

  p_data = KP_LIST_VIEW_PRIVATE_DATA (view);

  if (!p_data->date) 
    return;

  /* For all time, the log is needed to figure out which are the first
   * and the last date of the log */
  if (p_data->type == KP_VIEW_MODEL_TYPE_ALL_TIME
   && !KP_IS_TRAINING_LOG (p_data->log))
    g_return_if_reached ();
    
  kp_gui_get_dates_for_view_type (p_data->date, p_data->type, &p_data->date_s,
                                 &p_data->date_e, p_data->log);
}

static void
kp_list_view_update (KPListView *lv)
{
  kp_list_view_update_dates (lv);
  kp_list_view_update_entries (lv);
}

static void
kp_list_view_update_entries (KPListView *lv)
{
  KPListViewPrivateData *p_data;
  GtkTreeIter iter;
  gchar date_str[32];
  gchar *type_name;
  gchar *param_str;
  gchar *duration;
  gchar *comment;
  gchar *pace_str;
  guint32 pace;
  gchar *entry;
  GDate *date;
  GList *list;
  KPWorkout *wo;

  kp_debug ("Updating entries.");
  
  p_data = KP_LIST_VIEW_PRIVATE_DATA (lv);
  if (!p_data->log)
    return;
  
  list = kp_training_log_get_all_entries_between (p_data->log, p_data->date_s,
                                                  p_data->date_e, NULL);
  p_data->entries = list;
  gtk_list_store_clear (p_data->store);
 
  date = g_date_new ();
    
  pace_str = NULL;
  
  while (list) {
    entry = kp_calendar_entry_to_string (KP_CALENDAR_ENTRY (list->data));
    date = kp_calendar_time_get_date (KP_CALENDAR_ENTRY (list->data)->datetime);
    g_return_if_fail (g_date_valid (date));
   
    (void) g_date_strftime (date_str, sizeof (date_str)-1, "%x", date);

    type_name = kp_calendar_entry_get_human_name (list->data);

    gtk_list_store_append (p_data->store, &iter);
    gtk_list_store_set (p_data->store, &iter,
                        COLUMN_DATE, date_str,
                        COLUMN_TYPE, type_name,
                        COLUMN_DESCRIPTION, entry,
                        -1);
    
    if (KP_IS_WORKOUT (list->data)) {
      wo = KP_WORKOUT (list->data);
      
      if ((pace = kp_workout_get_pace (wo))) {
        /* Let's get rid of milliseconds*/
        pace /= 1000;
        pace *= 1000;
        
        pace_str = kp_date_mseconds_to_std_string (pace);
      } 

      param_str = kp_workout_get_params_string (wo);
      comment = kp_workout_get_param (wo, "comment");
      
      duration = kp_date_mseconds_to_std_string (
          kp_workout_get_duration_in_milliseconds (wo));
    
      g_return_if_fail (gtk_list_store_iter_is_valid (p_data->store, &iter));
      gtk_list_store_set (p_data->store, &iter,
                          COLUMN_TIME, duration,
                          COLUMN_SPORT, kp_workout_get_sport (wo),
                          COLUMN_OTHER, param_str,
                          COLUMN_PACE, (pace) ? pace_str : "",
                          COLUMN_DISTANCE, kp_workout_get_distance (wo),
                          COLUMN_COMMENT, comment, 
                          -1);

      g_free (param_str);
      g_free (duration);
      g_free (comment);
    }
    g_free (type_name);
    g_free (entry);

    list = list->next;
  }
}


static void
log_changed (KPTrainingLog *log, KPListView *lv)
{
  kp_list_view_update (lv);
}
  
static void
log_connect_signals (KPTrainingLog *log, KPListView *lv)
{
  g_signal_connect (G_OBJECT (log), "changed",
                    G_CALLBACK (log_changed), lv);
}

static void
log_disconnect_signals (KPTrainingLog *log, KPListView *lv)
{
  g_signal_handlers_disconnect_by_func (log, log_changed, lv);
}

static void
on_popup_delete_clicked (GtkWidget *menuitem, KPListView *lv)
{
  KPListViewPrivateData *p_data;
  KPCalendarEntry *entry;
  GtkTreeView *treeview = GTK_TREE_VIEW (lv);
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  GDate *date;
  gchar *date_str;
  gchar *mark;
  guint d,m,y;

  p_data = KP_LIST_VIEW_PRIVATE_DATA (lv);

  g_return_if_fail (GTK_IS_TREE_VIEW (treeview));
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  date = g_date_new ();
  
  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {

    gtk_tree_model_get (model, &iter,
                        COLUMN_DATE, &date_str,
                        COLUMN_DESCRIPTION, &mark,
                        -1);

    g_date_set_parse (date, date_str);

    if (!g_date_valid (date)) {
      kp_debug ("Couldn't parse date string: %s\n", date);
      return;
    }

    d = g_date_get_day (date);
    m = g_date_get_month (date);
    y = g_date_get_year (date);

    g_return_if_fail (KP_IS_TRAINING_LOG (p_data->log));
    
    entry = kp_training_log_get_entry (p_data->log, d, m, y, mark);

    if (KP_IS_CALENDAR_ENTRY (entry))
      kp_training_log_remove (p_data->log, entry);
  }
}


static void
on_popup_properties_clicked (GtkWidget *menuitem, KPListView *lv)
{
  KPListViewPrivateData *p_data;
  KPCalendarEntry *entry;
  GtkTreeView *treeview = GTK_TREE_VIEW (lv);
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkWidget *dialog;
  GtkTreeIter iter;
  GDate *date;
  gchar *date_str;
  gchar *mark;
  guint d,m,y;

  p_data = KP_LIST_VIEW_PRIVATE_DATA (lv);

  g_return_if_fail (GTK_IS_TREE_VIEW (treeview));
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  date = g_date_new ();
  
  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {

    gtk_tree_model_get (model, &iter,
                        COLUMN_DATE, &date_str,
                        COLUMN_DESCRIPTION, &mark,
                        -1);

    g_date_set_parse (date, date_str);

    if (!g_date_valid (date)) {
      kp_debug ("Couldn't parse date string: %s\n", date);
      return;
    }

    d = g_date_get_day (date);
    m = g_date_get_month (date);
    y = g_date_get_year (date);

    g_return_if_fail (KP_IS_TRAINING_LOG (p_data->log));
    
    entry = kp_training_log_get_entry (p_data->log, d, m, y, mark);

    if (KP_IS_CALENDAR_ENTRY (entry)) {
      dialog = kp_calendar_entry_info_dialog_new (entry);
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
    }
  }
}


static gboolean
on_button_press_event (GtkWidget *treeview, GdkEventButton *event,
                       gpointer data)
{
  GtkTreeSelection *selection;

  if (event->type != GDK_BUTTON_PRESS || event->button != 3)
    return FALSE;

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  view_popup_menu (treeview, event, data);

  return FALSE;
}


static gboolean
on_popup_menu (GtkWidget *treeview, gpointer data)
{
  view_popup_menu (treeview, NULL, data);
  return TRUE; 
}


static GtkWidget *
create_popup_menu (KPListView *lv)
{
  KPListViewPrivateData *p_data;
  GtkWidget *menu;

  p_data = KP_LIST_VIEW_PRIVATE_DATA (lv);

  GladeXML *xml = kp_gui_load ("popups", "listview_popup_menu");
  p_data->popup_mi_properties = KP_W (xml, "properties"); 
  p_data->popup_mi_delete = KP_W (xml, "delete");
  menu = KP_W (xml, "listview_popup_menu");
  
  g_signal_connect (G_OBJECT (p_data->popup_mi_delete), "activate",
                    G_CALLBACK (on_popup_delete_clicked), lv);
  g_signal_connect (G_OBJECT (p_data->popup_mi_properties), "activate",
                    G_CALLBACK (on_popup_properties_clicked), lv);

  g_object_unref (xml);
  return menu;
}

static void
view_popup_menu (GtkWidget *treeview, GdkEventButton *event, gpointer data)
{
  KPListViewPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gboolean empty = TRUE;
  gchar *entry;
  
  p_data = KP_LIST_VIEW_PRIVATE_DATA (treeview);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  g_return_if_fail (KP_IS_LIST_VIEW (treeview));
  /*g_return_if_fail (GTK_IS_TREE_SELECTION (selection));*/

  if (GTK_IS_TREE_SELECTION (selection)
  && gtk_tree_selection_get_selected (selection, &model, &iter)) {
    gtk_tree_model_get (model, &iter, COLUMN_DESCRIPTION, &entry, -1);
    
    kp_debug ("Selected: %s", entry);
   
    empty = FALSE;
  }
  
  gtk_widget_set_sensitive (p_data->popup_mi_properties, !empty);
  gtk_widget_set_sensitive (p_data->popup_mi_delete, !empty);
  
  gtk_menu_popup (GTK_MENU (p_data->popup_menu), NULL, NULL, NULL, NULL,
                  (event) ? event->button : 0,
                  gdk_event_get_time ((GdkEvent *) event));
}

