/*  Inti-GL: Integrated Foundation Classes
 *  Copyright (C) 2003 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

//! @file inti/gtk/gl/widget.h
//! @brief A C++ wrapper interface for the GtkGLExt widget functions.

#ifndef INTI_GTK_GL_WIDGET_H
#define INTI_GTK_GL_WIDGET_H

#ifndef INTI_G_TYPE_H
#include <inti/glib/type.h>
#endif
#ifndef __GTK_GL_WIDGET_H__
#include <gtk/gtkglwidget.h>
#endif

namespace Inti {

namespace Gdk {

namespace GL {
class Config;
class Context;
class Drawable;
class Window;
}

} // namespace Gdk

namespace Gtk {

namespace GL {

//! @class Widget widget.h inti/gdk/gl/widget.h
//! @brief A C++ wrapper interface for the GtkGLExt widget functions.
//!
//! Widget in an interface class that provides an OpenGL extension to Gtk::Widget API.

class Widget : virtual public G::TypeInstance
{
public:
//! @name Constructors
//! @{

	virtual ~Widget() = 0;
	//!< Destructor.

//! @}
//! @name Accessors
//! @{

	bool is_gl_capable() const;
	//!< Determines whether the widget is OpenGL-capable.
	//!< @return <EM>true</EM> if the widget is OpenGL-capable, <EM>false</EM> otherwise.

	Gdk::GL::Config* get_gl_config() const;
	//!< Gets the Gdk::GL::Config referred to by the widget.
	//!< @return The Gdk::GL::Config.

	Gdk::GL::Context* get_gl_context() const;
	//!< Gets the Gdk::GL::Context with the appropriate Gdk::GL::Drawable for the widget.
	//!< @return The Gdk::GL::Context.
	//!<
	//!< <BR>A Gdk::GL::Context is needed for the method Gdk::GL::Drawable::begin, or
	//!< for sharing display lists (see set_gl_capability()).

	Gdk::GL::Window* get_gl_window() const;
	//!< Gets the Gdk::GL::Window owned by the widget.
	//!< @return The Gdk::GL::Window.

	Gdk::GL::Drawable* get_gl_drawable() const;
	//!< Gets the Gdk::GL::Drawable owned by the widget.
	//!< @return The Gdk::GL::Drawable.
	//!<
	//!< <BR>This method returns the result of get_gl_window() cast to a Gdk::GL::Drawable.

//! @}
//! @name Methods
//! @{

	Pointer<Gdk::GL::Context> create_gl_context(bool direct = true, int render_type = GDK_GL_RGBA_TYPE);
	//!< Creates a new Gdk::GL::Context with the appropriate Gdk::GL::Drawable for this widget.
	//!< @param direct Whether rendering is to be done with a direct connection to the graphics system.
	//!< @param render_type Gdk::GL::RGBA_TYPE or Gdk::GL::COLOR_INDEX_TYPE (currently not used).
	//!< @return The new Context, or null if an error occurs.
	//!<
	//!< <BR>See also get_gl_context().

	Pointer<Gdk::GL::Context> create_gl_context(const Gdk::GL::Context& share_list, bool direct = true, int render_type = GDK_GL_RGBA_TYPE);
	//!< Creates a new Gdk::GL::Context with the appropriate Gdk::GL::Drawable for this widget.
	//!< @param share_list The Gdk::GL::Context with which to share display lists and texture objects.
	//!< @param direct Whether rendering is to be done with a direct connection to the graphics system.
	//!< @param render_type Gdk::GL::RGBA_TYPE or Gdk::GL::COLOR_INDEX_TYPE (currently not used).
	//!< @return The new Context, or null if an error occurs.
	//!<
	//!< <BR>See also get_gl_context().

	bool set_gl_capability(const Gdk::GL::Config& glconfig, bool direct = true, int render_type = GDK_GL_RGBA_TYPE);
	//!< Sets the OpenGL-capability for the widget.
	//!< @param glconfig A Gdk::GL::Config.
	//!< @param direct Whether rendering is to be done with a direct connection to the graphics system.
	//!< @param render_type Gdk::GL::RGBA_TYPE or Gdk::GL::COLOR_INDEX_TYPE (currently not used).
	//!< @return <EM>true</EM> if it is successful, <EM>false</EM> otherwise.
	//!<
	//!< <BR>This method prepares the widget for its use with OpenGL. No sharing of
	//!< display lists and texture objects takes place when you call this method.

	bool set_gl_capability(const Gdk::GL::Config& glconfig, const Gdk::GL::Context& share_list, bool direct = true, int render_type = GDK_GL_RGBA_TYPE);
	//!< Sets the OpenGL-capability for the widget.
	//!< @param glconfig A Gdk::GL::Config.
	//!< @param share_list The Gdk::GL::Context with which to share display lists and texture objects.
	//!< @param direct Whether rendering is to be done with a direct connection to the graphics system.
	//!< @param render_type Gdk::GL::RGBA_TYPE or Gdk::GL::COLOR_INDEX_TYPE (currently not used).
	//!< @return <EM>true</EM> if it is successful, <EM>false</EM> otherwise.
	//!<
	//!< <BR>This method prepares the widget for its use with OpenGL.

//! @}
};

} // namespace GL

} // namespace Gtk

} // namespace Inti

#endif // INTI_GTK_GL_WIDGET_H

