/*  Coolwave example.
 *
 *  coolwave.cc
 *
 *  Inti Translation of the original GtkGLExt coolwave example.
 *  Copyright (c) 2002 Alif Wahid <awah005@users.sourceforge.net>
 *  Modified by Naofumi Yasufuku  <naofumi@users.sourceforge.net>
 *
 *  Coolwave is a modified version of the old IrisGL demo 'newave',
 *  first ported to OpenGL and Glut by Erik Larsen. It was modified
 *  it to use Gtk and GtkGLExt comprehensively along with ofcourse
 *  OpenGL. Now it has been translated into C++ using Inti.
 */

#include "coolwave.h"
#include <inti/gdk/keyval.h>
#include <inti/gdk/gl/drawable.h>
#include <inti/gtk/button.h>
#include <inti/gtk/menuitem.h>
#include <inti/gtk/gl/init.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdkglglext.h> // use OpenGL extensions
#include <GL/gl.h>
#include <GL/glu.h>
#include <iostream>
#include <cstdio>
#include <cstdlib>
#include <cmath>

namespace { // Waving functions

const int TIMEOUT_INTERVAL = 10;
const int MAXGRID = 63;
const float SQRTOFTWOINV = 1.0 / 1.414213562;
const int grid = MAXGRID / 2;
float force[MAXGRID][MAXGRID];
float veloc[MAXGRID][MAXGRID];
float posit[MAXGRID][MAXGRID];
const float dt = 0.008;
const float zNear = grid / 10.0;
const float zFar = grid * 3.0;
const float lightPosition[4] = {0.0, 0.0, 1.0, 1.0};

/*  get_force
 */

void get_force()
{
	for (int i = 0; i < grid; i++)
	{
		for(int j = 0; j < grid; j++)
		{
			force[i][j] = 0.0;
		}
	}

	for (int i = 2; i < grid - 2; i++)
	{
		for (int j = 2; j < grid - 2; j++)
		{
			float d = posit[i][j] - posit[i][j - 1];
			force[i][j] -= d;
			force[i][j-1] += d;

			d = posit[i][j] - posit[i - 1][j];
			force[i][j] -= d;
			force[i - 1][j] += d;

			d = (posit[i][j] - posit[i][j + 1]);
			force[i][j] -= d ;
			force[i][j + 1] += d;

			d = (posit[i][j] - posit[i + 1][j]);
			force[i][j] -= d ;
			force[i + 1][j] += d;

			d = (posit[i][j] - posit[i + 1][j + 1]) * SQRTOFTWOINV;
			force[i][j] -= d ;
			force[i + 1][j + 1] += d;

			d = (posit[i][j] - posit[i - 1][j - 1]) * SQRTOFTWOINV;
			force[i][j] -= d ;
			force[i - 1][j - 1] += d;

			d = (posit[i][j] - posit[i + 1][j - 1]) * SQRTOFTWOINV;
			force[i][j] -= d ;
			force[i + 1][j - 1] += d;

			d = (posit[i][j] - posit[i - 1][j + 1]) * SQRTOFTWOINV;
			force[i][j] -= d ;
			force[i - 1][j + 1] += d;
		}
	}
}

/*  get_velocity()
 */

void get_velocity()
{
	for (int i = 0; i < grid; i++)
	{
		for (int j = 0; j < grid; j++)
			veloc[i][j] += force[i][j] * dt;
	}
}

/*  get_position
 */

void get_position (void)
{
	for (int i = 0; i < grid; i++)
	{
		for (int j = 0; j < grid; j++)
			posit[i][j] += veloc[i][j];
	}
}

/*  draw_wire_frame
 */

void draw_wireframe()
{
	glColor3f(1.0, 1.0, 1.0);

	for (int i = 0; i < grid; i++)
	{
		glBegin(GL_LINE_STRIP);
		for(int j = 0; j < grid; j++)
			glVertex3f((float)i, (float)j, (float)posit[i][j]);
		glEnd();
	}

	for (int i = 0; i < grid; i++)
	{
		glBegin(GL_LINE_STRIP);
		for(int j = 0; j < grid; j++)
			glVertex3f((float)j, (float)i, (float)posit[j][i]);
		glEnd();
	}
}

/*  reset_wire_frame
 */

void reset_wireframe()
{
	for (int i = 0; i < grid; i++)
	{
		for (int j = 0; j < grid; j++)
		{
			force[i][j] = 0.0;
			veloc[i][j] = 0.0;

			posit[i][j] = (sin(G_PI * 2 * ((float)i / (float)grid)) +
			               sin(G_PI * 2 * ((float)j / (float)grid))) * grid / 6.0;

			if (i == 0 || j == 0 || i == grid - 1 || j == grid - 1)
				posit[i][j] = 0.0;
		}
	}
}

} // namespace

/* GLDrawingArea
 */

GLDrawingArea::GLDrawingArea(Gdk::GL::Config *glconfig)
: Gtk::GL::DrawingArea(*glconfig, 300, 200)
{
	animate = true;
	beginX = beginY = 0.0;
	sphi = 90.0;
	stheta = 45.0;
	sdepth = 5.0 / 4.0 * (MAXGRID / 2);
	aspect = 5.0 / 4.0;
}

GLDrawingArea::~GLDrawingArea()
{
}

void
GLDrawingArea::on_realize()
{
	// All the OpenGL initialization should be performed here,
	// such as default background colour, certain states etc.

	Gtk::GL::DrawingArea::on_realize();
	Gdk::GL::Drawable *gldrawable = get_gl_drawable();

	// OpenGL begin
	if (!gldrawable->gl_begin(*get_gl_context()))
		return;

	// glPolygonOffsetEXT
	GdkGLProc proc = gdk_gl_get_glPolygonOffsetEXT();
	if (!proc)
	{
		// glPolygonOffset
		proc = gdk_gl_get_proc_address ("glPolygonOffset");
		if (!proc)
		{
			std::cout << "Sorry, glPolygonOffset() is not supported by this renderer.\n";
			Main::quit();
		}
	}

	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LEQUAL);
	glClearColor(0.0, 0.0, 0.0, 0.0);
	gdk_gl_glPolygonOffsetEXT(proc, 1.0, 1.0);
	glEnable(GL_CULL_FACE);
	glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);
	glHint(GL_POLYGON_SMOOTH_HINT, GL_NICEST);
	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
	glEnable(GL_COLOR_MATERIAL);
	glColorMaterial(GL_FRONT, GL_DIFFUSE);
	glLightfv(GL_LIGHT0, GL_POSITION, lightPosition);
	glEnable(GL_LIGHT0);
	glShadeModel(GL_FLAT);
	glDisable(GL_LIGHTING);

	reset_wireframe ();

	gldrawable->gl_end();
	// OpenGL end
}

bool
GLDrawingArea::on_configure_event(const Gdk::EventConfigure& event)
{
	// Any processing required when the OpenGL-capable drawing area is
	// re-configured should be done here. Almost always it will be used
	// to resize the OpenGL viewport when the window is resized.

	Gdk::GL::Drawable *gldrawable = get_gl_drawable();

	float width = get_allocation().width();
	float height = get_allocation().height();

	// OpenGL begin
	if (!gldrawable->gl_begin(*get_gl_context()))
		return false;

	aspect = width / height;
	glViewport(0, 0, (int)width, (int)height);
	gldrawable->gl_end();
	// OpenGL end

	return true;
}

bool
GLDrawingArea::on_expose_event(const Gdk::EventExpose& event)
{
	// All the OpenGL re-drawing should be done here. This is repeatedly called
	// as the painting routine every time the 'expose'/'draw' event is signalled.

	Gdk::GL::Drawable *gldrawable = get_gl_drawable();

	// OpenGL begin
	if (!gldrawable->gl_begin(*get_gl_context()))
		return false;

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(64.0, aspect, zNear, zFar);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	glTranslatef(0.0,0.0,-sdepth);
	glRotatef(-stheta, 1.0, 0.0, 0.0);
	glRotatef(sphi, 0.0, 0.0, 1.0);
	glTranslatef(-(float)((grid + 1) / 2 - 1), -(float)((grid + 1) / 2 - 1), 0.0);

	draw_wireframe ();

	if (gldrawable->is_double_buffered())
		gldrawable->swap_buffers();
	else
		glFlush ();

	glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	gldrawable->gl_end();
	// OpenGL end

	return true;
}

bool
GLDrawingArea::on_motion_notify_event(const Gdk::EventMotion& event)
{
	// Any processing required when the OpenGL-capable drawing area
	// is under drag motion should be done here.

	bool redraw = false;

	if (event.state() & Gdk::BUTTON1_MASK)
	{
		sphi += (event.x() - beginX) / 4.0;
		stheta += (beginY - event.y()) / 4.0;
		redraw = true;
	}

	if (event.state() & Gdk::BUTTON2_MASK)
	{
		sdepth -= ((event.y() - beginY) / (get_allocation().height())) * (MAXGRID / 2);
		redraw = true;
	}

	beginX = event.x();
	beginY = event.y();

	if (redraw && !animate)
		queue_draw();

	return true;
}

bool
GLDrawingArea::on_button_press_event(const Gdk::EventButton& event)
{
	// Any processing required when the mouse buttons are pressed
	// on the OpenGL-capable drawing area should be done here.

	switch (event.button())
	{
	case 1: // left mouse button
	case 2: // middle mouse button
		beginX = event.x();
		beginY = event.y();
		return true;

	case 3: // right mouse button
	{
		PopupMenu menu(*this);
		menu.popup(event.button());
		return true;
	}

	default:
		return false;
	}
}

bool
GLDrawingArea::on_key_press_event(const Gdk::EventKey& event)
{
	// Any processing required when key presses occur should be done here.

	switch (event.keyval())
	{
	case GDK_r:
		on_init_wireframe();
		break;

	case GDK_a:
		on_toggle_animation();
		break;

	case GDK_w:
		if (!animate)
			on_timeout();
		break;

	case GDK_plus:
		sdepth -= 2.0;
		break;

	case GDK_minus:
		sdepth += 2.0;
		break;

	case GDK_Escape:
		Main::quit();
		break;

	default:
		return false;
	}

	if (!animate)
		queue_draw();

	return true;
}

bool
GLDrawingArea::on_map_event(const Gdk::EventAny& event)
{
	if (animate)
		timeout_connect();
	return true;
}

bool
GLDrawingArea::on_unmap_event(const Gdk::EventAny& event)
{
	timeout_disconnect();
	return true;
}

bool
GLDrawingArea::on_visibility_notify_event(const Gdk::EventVisibility& event)
{
	if (animate)
	{
		if (event.state() == Gdk::VISIBILITY_FULLY_OBSCURED)
			timeout_disconnect();
		else
			timeout_connect();
	}
	return true;
}

bool
GLDrawingArea::on_timeout()
{
	// Often in animations, timeout functions are suitable for continous frame updates.
	get_force();
	get_velocity();
	get_position();
	queue_draw();
	return true;
}

void
GLDrawingArea::on_init_wireframe()
{
	reset_wireframe();
	queue_draw();
}

void
GLDrawingArea::on_toggle_animation()
{
	animate = !animate;

	if (animate)
	{
		timeout_connect();
	}
	else
	{
		timeout_disconnect();
		queue_draw();
	}
}

void
GLDrawingArea::timeout_connect()
{
	if (!timeout_connection.is_connected())
		timeout_connection = Main::timeout_signal.connect(slot(this, &GLDrawingArea::on_timeout), TIMEOUT_INTERVAL);
}

void
GLDrawingArea::timeout_disconnect()
{
	if (timeout_connection.is_connected())
		timeout_connection.disconnect();
}

/*  PopupMenu
 */

PopupMenu::PopupMenu(GLDrawingArea& drawing_area)
{
	// Toggle animation
	Gtk::MenuItem *item = new Gtk::MenuItem("Toggle Animation");
	append(*item, slot(drawing_area, &GLDrawingArea::on_toggle_animation));

	// Initialize wireframe model
	item = new Gtk::MenuItem("Initialize");
	append(*item, slot(drawing_area, &GLDrawingArea::on_init_wireframe));

	// Quit
	item = new Gtk::MenuItem("Quit");
	append(*item, slot(&Main::quit));

	show_all();
}

PopupMenu::~PopupMenu()
{
}

/*  Window
 */

Window::Window(Gdk::GL::Config *glconfig)
{
	set_title("CoolWave");

	// Perform the resizes immediately.
	set_resize_mode(Gtk::RESIZE_IMMEDIATE);

	// Get automatically redrawn if any of their children changed allocation.
	set_reallocate_redraws(true);

	Gtk::VBox *vbox = new Gtk::VBox;
	add(*vbox);
	vbox->show();

	// Drawing area to draw OpenGL scene.
	GLDrawingArea *drawing_area = new GLDrawingArea(glconfig);
	drawing_area->add_events(Gdk::BUTTON1_MOTION_MASK | Gdk::BUTTON2_MOTION_MASK | Gdk::BUTTON_PRESS_MASK | Gdk::VISIBILITY_NOTIFY_MASK);
	vbox->pack_start(*drawing_area);
	drawing_area->show();

	// Simple quit button.
	Gtk::Button *button = new Gtk::Button("Quit");
	button->sig_clicked().connect(slot(&Main::quit));
	vbox->pack_start(*button, false, false);
	button->show();
	show();
}

Window::~Window()
{
}

int main (int argc, char *argv[])
{
	using namespace Main;

	init(&argc, &argv);

	// Initialize GtkGLExt
	Gtk::GL::init(&argc, &argv);

	// Configure OpenGL-capable visual.
	Pointer<Gdk::GL::Config> glconfig = Gdk::GL::Config::create(Gdk::GL::MODE_RGB | Gdk::GL::MODE_DEPTH | Gdk::GL::MODE_DOUBLE);
	if (!glconfig)
	{
		std::cout << "*** Cannot find the double-buffered visual." << std::endl;
		std::cout << "*** Trying single-buffered visual." << std::endl;

		// Try single-buffered visual
		glconfig = Gdk::GL::Config::create(Gdk::GL::MODE_RGB | Gdk::GL::MODE_DEPTH);
		if (!glconfig)
		{
			std::cout << "*** No appropriate OpenGL-capable visual found." << std::endl;
			return 1;
		}
	}

	Window window(glconfig);
	window.sig_destroy().connect(slot(&Inti::Main::quit));

	run();
	return 0;
}

