//$Id: session.cc,v 1.21 2004/03/20 23:46:03 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2003 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "session.h"

#include "config.h"

#include <gtk--/main.h>
#include <glade/glade.h>
#include <gnome--/client.h>
#include <libgnome/gnome-config.h>
#include <gconf/gconf.h>
#include <libgnomevfs/gnome-vfs.h>

#include <sigc++/retbind.h>
#include "queued-signal.h"

#include "resource-manager.h"
#include "mainwin.h"
#include "ui.h"
#include "preferences.h"

#include "io/xml-loader.h"
#include "io/xml-saver.h"

using namespace Guikachu;

namespace {
    void io_load_error (const std::string &uri, const std::string &error_msg);
    void io_save_error (const std::string &uri, const std::string &error_msg);

    void gtk_queuer (SigC::Slot0<void> slot)
    {
	Gtk::Main::idle.connect (SigC::retbind (slot, 0));
    }
} // anonymous namespace

Main* Guikachu::Main::instance_ = 0;

Main::Main (int argc, char **argv):
    Gnome::Main (PACKAGE, VERSION, argc, argv, NULL, 0, &ctx),
    cmd_self (argv[0])
{
    if (instance_)
    {
	g_warning("Guikachu::Main::Main called twice!\n");
	return;
    }

    instance_ = this;

    gconf_init (argc, argv, 0);
    glade_gnome_init ();
    gnome_vfs_init ();
    QueuedSignal::set_signal_queuer (gtk_queuer);
    
    // Set up session managment stuff
    Gnome::Client* session_client = Gnome::Client::master_client ();
    if (session_client)
    {
	session_client->die.connect (SigC::slot(this, &Main::session_die_cb));
	session_client->save_yourself.connect (SigC::slot(this, &Main::session_save_cb));
    }

    // Store command-line arguments
    const char **args = poptGetArgs (ctx);
    if (args)
	for (; *args; args++)
	    cmd_arguments.push_back (*args);
    poptFreeContext (ctx);

    // Init preferences
    Preferences::init ();

    // Create initial document
    doc = new ResourceManager;
    
    // Create main window
    mainwin = new GUI::MainWin;
    mainwin->set_manager (doc);

    // Load document (last filename) from command line
    if (cmd_arguments.size ())
    {
	std::string uri = IO::create_canonical_uri (*cmd_arguments.rbegin ());
	load_doc (uri, "application/x-guikachu");
    }
}


Main::~Main ()
{
    // Shut down preferences
    Preferences::shutdown ();
}

void Main::run_impl ()
{
    mainwin->show ();
    Gnome::Main::run_impl ();
}

namespace {

void io_load_error (const std::string &uri, const std::string &error_msg)
{    
    char *buffer = g_strdup_printf (
	_("Unable to open `%s':\n%s"),
	UI::visible_filename (uri).c_str () , error_msg.c_str ());

    UI::show_error (buffer);
    g_free (buffer);
}

void io_save_error (const std::string &uri, const std::string &error_msg)
{    
    char *buffer = g_strdup_printf (
	_("Unable to save `%s':\n%s"),
	UI::visible_filename (uri).c_str (), error_msg.c_str ());

    UI::show_error (buffer);
    g_free (buffer);
}

} // anonymous namespace

void Main::set_doc (ResourceManager *new_doc)
{
    delete doc;
    doc = new_doc;

    mainwin->set_manager (doc);
}

void Main::new_doc ()
{
    delete doc;
    doc = new ResourceManager;
    
    mainwin->set_manager  (doc);
    mainwin->set_filename ("");
}

void Main::save_doc (const std::string &filename)
{
    std::string uri = IO::create_canonical_uri (filename);
    
    IO::Saver    *saver = new IO::XMLSaver;
    IO::IOResult  res = saver->save (doc, uri);
    delete saver;

    if (res == GNOME_VFS_OK)
    {
	doc->clear_dirty ();
	set_filename (uri, "application/x-guikachu");
    } else {
	io_save_error (uri, gnome_vfs_result_to_string (res));
    }
}

void Main::load_doc (const std::string &filename,
		     const std::string &mime_type)
{
    std::string uri = IO::create_canonical_uri (filename);
    
    ResourceManager *new_doc = new ResourceManager;

    IO::Loader   *loader = IO::IOFactory::instance ()->create_loader (mime_type);
    if (!loader)
    {
	char *error_msg = g_strdup_printf (_("File format %s unknown"), mime_type.c_str ());
	io_load_error (uri, error_msg);
	g_free (error_msg);

	delete new_doc;
	return;
    }
    
    IO::IOResult res = loader->load (new_doc, uri);
    delete loader;

    if (res == GNOME_VFS_OK)
    {
	set_doc (new_doc);
	set_filename (uri, mime_type);
    } else {
	io_load_error (uri, gnome_vfs_result_to_string (res));
	delete new_doc;
    }
}

void Guikachu::Main::session_die_cb ()
{
    quit ();
}

int Guikachu::Main::session_save_cb (int                phase,
				     GnomeSaveStyle     save_style,
				     gint               is_shutdown,
				     GnomeInteractStyle interact_style,
				     int                is_fast)
{
    save_open_file ();
    
    return TRUE;
}

void Guikachu::Main::save_open_file ()
{
    Gnome::Client* session_client = Gnome::Client::master_client ();
    if (session_client->is_connected ())
    {
	std::vector<std::string> argv;
	argv.push_back (cmd_self);
	argv.push_back (last_filename);
	
	session_client->set_clone_command (argv);
	session_client->set_restart_command (argv);
    }
}

void Guikachu::Main::set_filename (const std::string &filename,
				   const std::string &mime_type)
{
    if (filename != "")
	Preferences::Interface::add_recent_file (
	    Preferences::Interface::RecentFileEntry (filename, mime_type));

    if (mime_type == "application/x-guikachu")
    {
	last_filename = filename;
	mainwin->set_filename (filename);
    } else {
	mainwin->set_filename ("");
    }
}
