/*  cssed (c) Iago Rubio 2003, 2004 - A tiny CSS editor.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <stdio.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <gmodule.h>

#include "cssedwindow.h"
#include "document.h"
#include "interface.h"
#include "cssdialogs-class.h"
#include "plugin.h"
#include "support.h"
#include "utils.h"
#include "debug.h"

#define PRIV(obj) ((obj)->priv)

struct _CssedPluginPriv
{
	GModule* module_instance;	
	CssedWindow* window;
	gchar* filename;
	gboolean loaded;
};
/* debugging function */
void
dump_list (gpointer data,  gpointer user_data)
{
	CssedPlugin* p;
	
	p = (CssedPlugin*) data;
	
	g_print( "dump_list:\n" );
	if( p == NULL ){
		g_print( "Null plugin\n" );
		return;
	}else{
		g_print( "Plugin %p\n", p);
	}
	g_print("Name %s\n",  p->name );
	if( PRIV(p)->filename == NULL ){
		g_print( "Plugin filename NULL\n");
		return;
	}else{
		g_print( "Plugin filename %s\n", PRIV(p)->filename);
	}
}

CssedPlugin*
cssed_init_plugin(CssedWindow* window, gchar* filename)
{
	GModule* module = NULL;
	CssedPlugin *plugin = NULL;
	CssedPlugin *(*init_plugin)() = NULL;
	gboolean test;
	gchar* errmsg;
	
   	module = g_module_open( filename, 0 );	
	
	if( module == NULL ){
		errmsg = g_strdup_printf(_("Cannot open module \"%s\"\n\n%s\n"),filename, g_module_error() );
		g_print("%s\n", g_module_error());
		cssed_error_message( errmsg, _("Cannot load plugin") );
		g_free( errmsg );
		return NULL;		
	}

	test = g_module_symbol( module, "init_plugin", (gpointer*) &init_plugin);

	if (!test)
	{
		errmsg = g_strdup_printf(_("Error while loading \"%s\"\n\n%s\n"),filename, g_module_error() );
		cssed_error_message( errmsg, _("Cannot load plugin") );
		g_free( errmsg );
		g_module_close(module);
		return NULL;
	}
	
	plugin = (*init_plugin)();
	DBGMSG ("cssed_init_plugin address: %p\n", plugin);

	if (!plugin)
	{
		errmsg = g_strdup_printf(_("Error while initializing: \"%s\"\n"),filename);
		cssed_error_message( errmsg, _("Cannot load plugin") );
		g_free( errmsg );		
		g_module_close(module);
		return NULL;
	}

	PRIV(plugin) = g_malloc0 (sizeof (CssedPluginPriv)) ;
	PRIV(plugin)->module_instance = (gpointer) module;
	PRIV(plugin)->window = window;
	PRIV(plugin)->filename = g_strdup( filename );
	PRIV(plugin)->loaded = FALSE;
	return plugin;	
}

void
cssed_load_plugin( CssedPlugin* plugin )
{
	CssedWindow* window;
	
	if(	plugin->load_plugin(plugin) == TRUE ){
		PRIV(plugin)->loaded = TRUE;
		window = PRIV(plugin)->window;
		cssed_window_add_plugin_to_list( window,(gpointer)  plugin );
	}else{
		cssed_unload_plugin( plugin );
		cssed_error_message(_("The plugin returned an internal error"),_("Cannot load plugin"));
	}
}

void
cssed_unload_plugin( CssedPlugin* plugin )
{
	GModule* module;
	gboolean closed;
	CssedPluginPriv* priv;
	CssedWindow* window;
	
	g_return_if_fail( plugin != NULL );
	
	plugin->clean_plugin( plugin );	
	module = PRIV(plugin)->module_instance;
	window = PRIV(plugin)->window;
	priv = PRIV(plugin);	

	if( module != NULL ){
		g_free( priv->filename );
		g_free( priv );
		cssed_window_delete_plugin_from_list ( window, (gpointer)  plugin );
		closed =  g_module_close ( module );
		if(!closed){
			cssed_error_message( _("Cannot unload the plugin"), 
								 _("Unable to unload") );
		}
		return;
	}else{
		return;
	}
}

gboolean
cssed_is_plugin_file_loaded ( CssedWindow* window, gchar* path )
{
	GList* plugin_list;
	gchar* filename;
	
	plugin_list = cssed_window_get_plugin_list_first(  window );		

	while( plugin_list ){
		CssedPlugin* current_plugin;
		current_plugin = (CssedPlugin*) plugin_list->data;
		filename = cssed_plugin_get_filename( current_plugin );
		if( filename != NULL ){
			if( strcmp( path, filename ) == 0 ){
				return TRUE;
			}
		}
		plugin_list = g_list_next( plugin_list );
	}	
	return FALSE;
}

CssedPlugin* 
cssed_get_plugin_if_file_loaded ( CssedWindow* window, gchar* path )
{
	GList* plugin_list;
	gchar* filename;
	
	plugin_list  = cssed_window_get_plugin_list_first(  window );	

	while( plugin_list ){
		CssedPlugin* current_plugin;
		current_plugin = (CssedPlugin*) plugin_list->data;
		filename = cssed_plugin_get_filename(current_plugin);
		if( filename != NULL ){
			if( strcmp( path, filename ) == 0 ){
				return current_plugin;
			}
		}
		plugin_list = g_list_next( plugin_list );
	}	
	return NULL;
}

gchar*
cssed_plugin_get_filename( CssedPlugin* plugin )
{
	g_return_val_if_fail( plugin != NULL, NULL );
	g_return_val_if_fail( PRIV(plugin)->filename != NULL, NULL );
	return  PRIV(plugin)->filename;
}
// callback
void
cssed_plugin_unload_callback( GtkWidget* widget, gpointer plugin )
{
	cssed_unload_plugin( (CssedPlugin*) plugin );
}

// UI functions
gboolean
cssed_add_menu_item_to_plugins_menu( CssedPlugin* plugin, GtkWidget* menu_item )
{
	GtkWidget* plugins_menu;
	GtkWidget* window_widget;
	CssedWindow* window;
	
	window = PRIV(plugin)->window;
	window_widget = cssed_window_get_window_widget( window );
	
	if( !GTK_IS_WIDGET( window_widget ) ){
		return FALSE;
	}
	plugins_menu = lookup_widget( window_widget, "plugins_menu");
	if( plugins_menu == NULL ){
		return FALSE;
	}

	gtk_container_add(GTK_CONTAINER(plugins_menu), menu_item);		
	return TRUE;
}

gboolean
cssed_plugin_add_widget_to_toolbar( CssedPlugin* plugin, 
							 GtkWidget* widget,
							 const char *tooltip_text,
							 const char *tooltip_private_text
							)
{
	GtkWidget* toolbar;
	GtkWidget* window_widget;
	CssedWindow* window;
	
	window = PRIV(plugin)->window;
	window_widget = cssed_window_get_window_widget( window );
	
	if( !GTK_IS_WIDGET( window_widget ) ){
		return FALSE;
	}
	toolbar = lookup_widget( window_widget, "toolbar_tools");
	if( toolbar == NULL ){
		return FALSE;
	}
	gtk_toolbar_append_widget (	GTK_TOOLBAR(toolbar), widget, 
								tooltip_text, tooltip_private_text );
	
	return TRUE;	
}
							
gboolean
cssed_plugin_add_page_with_widget_to_footer( CssedPlugin* plugin,
											GtkWidget* widget,
											gchar* label_text )
{
	GtkWidget* footer_notebook;
	GtkWidget* window_widget;
	GtkWidget* label;
	CssedWindow* window;
	
	if( plugin == NULL )
		return FALSE;
	
	window = PRIV(plugin)->window;
	window_widget = cssed_window_get_window_widget( window );	
	footer_notebook = cssed_window_get_footer_notebook (  window );
	
	label = gtk_label_new( label_text );
	
	gtk_notebook_append_page   (GTK_NOTEBOOK(footer_notebook),
                                widget,
                                label);
	return TRUE;
}

gboolean
cssed_plugin_remove_page_with_widget_in_footer( CssedPlugin* plugin,
											GtkWidget* widget )
{
	GtkWidget* footer_notebook;
	GtkWidget* window_widget;
	gint page_index;
	CssedWindow* window;
	
	if( plugin == NULL )
		return FALSE;
	
	window = PRIV(plugin)->window;
	window_widget = cssed_window_get_window_widget( window );	
	footer_notebook = cssed_window_get_footer_notebook ( window );	
	
	page_index = gtk_notebook_page_num  (GTK_NOTEBOOK(footer_notebook),
                                         widget);
	if( page_index != -1 )
	{
		gtk_notebook_remove_page (GTK_NOTEBOOK(footer_notebook), page_index);
		return TRUE;
	}
	else
	{
		return FALSE;
	}		
}

gboolean
cssed_plugin_add_page_with_widget_to_sidebar( CssedPlugin* plugin,
											GtkWidget* widget,
											gchar* label_text )
{
	GtkWidget* sidebar_notebook;
	GtkWidget* label;
	CssedWindow* window;
	
	if( plugin == NULL )
		return FALSE;
	
	window = PRIV(plugin)->window;
	sidebar_notebook = cssed_window_get_sidebar_notebook ( window );
	
	label = gtk_label_new( label_text );
	
	gtk_notebook_append_page   (GTK_NOTEBOOK(sidebar_notebook),
                                widget,
                                label);
	return TRUE;
}

gboolean
cssed_plugin_remove_page_with_widget_in_sidebar( CssedPlugin* plugin,
											GtkWidget* widget )
{
	GtkWidget* sidebar_notebook;
	gint page_index;
	CssedWindow* window;
	
	if( plugin == NULL )
		return FALSE;
	
	window = PRIV(plugin)->window;
	sidebar_notebook = cssed_window_get_sidebar_notebook ( window );	
	
	page_index = gtk_notebook_page_num  (GTK_NOTEBOOK(sidebar_notebook),
                                         widget);
	if( page_index != -1 )
	{
		gtk_notebook_remove_page        (GTK_NOTEBOOK(sidebar_notebook),
                                           page_index);
		return TRUE;
	}
	else
	{
		return FALSE;
	}		
}

gboolean
cssed_plugin_select_page_with_widget_in_sidebar( CssedPlugin* plugin,
											GtkWidget* widget )
{
	GtkWidget* sidebar_notebook;
	gint page_index;
	CssedWindow* window;
	
	if( plugin == NULL )
		return FALSE;
	
	window = PRIV(plugin)->window;
	sidebar_notebook = cssed_window_get_sidebar_notebook ( window );	

	page_index = gtk_notebook_page_num  (GTK_NOTEBOOK(sidebar_notebook),
                                         widget);
	if( page_index != -1 )
	{
		gtk_notebook_set_current_page     (GTK_NOTEBOOK(sidebar_notebook),
                                           page_index);
		return TRUE;
	}
	else
	{
		return FALSE;
	}

}

gboolean
cssed_plugin_select_page_with_widget_in_footer( CssedPlugin* plugin,
											GtkWidget* widget )
{
	GtkWidget* footer_notebook;
	gint page_index;
	CssedWindow* window;
	
	if( plugin == NULL )
		return FALSE;
	
	window = PRIV(plugin)->window;
	footer_notebook = cssed_window_get_footer_notebook ( window );	

	page_index = gtk_notebook_page_num  (GTK_NOTEBOOK(footer_notebook),
                                         widget);
	if( page_index != -1 )
	{
		gtk_notebook_set_current_page     (GTK_NOTEBOOK(footer_notebook),
                                           page_index);
		return TRUE;
	}
	else
	{
		return FALSE;
	}
}

/* ***************** utility dialogs ************************* */
// show an error dialog
void
cssed_plugin_error_message( gchar* message, gchar* title )
{
	cssed_error_message (message, title);
}
// this returns a boolean with the user's response OK = TRUE, !OK = FALSE
gboolean
cssed_plugin_confirm_dialog ( gchar* message, gchar* title )
{
	GtkWidget* dialog;
	gint response;
	
	dialog = create_ok_cancel_dialog (title, message);
	response = gtk_dialog_run( GTK_DIALOG(dialog) );
	
	if( response == GTK_RESPONSE_OK ){
		return TRUE;
	}else{
		return FALSE;
	}
}
/* ***************** document functions ********************** */
// open a named file
void
cssed_plugin_open_file( CssedPlugin* plugin, gchar* filename )
{
	CssedWindow* window;

	g_return_if_fail( plugin != NULL );
	
	window = PRIV(plugin)->window;
	create_and_attach_new_named_doc( window, filename );
}
// shows the open file selector dialog
void
cssed_plugin_open_file_with_fileselector( CssedPlugin* plugin )
{
	GtkWidget* opensel;
	CssedWindow* window;
#ifdef GTK_ATLEAST_2_4
	gchar* last_open_dir;
	gchar* filename;
	gchar* fullpath;
#endif

	g_return_if_fail( plugin != NULL );
	window = PRIV(plugin)->window;		

#ifndef GTK_ATLEAST_2_4
	opensel = create_open_fileselection (window);
	if (opensel != NULL)
	{
		gtk_widget_show (opensel);
	}
#else 
	opensel = gtk_file_chooser_dialog_new(_("Select file"), NULL,
												GTK_FILE_CHOOSER_ACTION_OPEN,
												GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
												GTK_STOCK_OPEN, GTK_RESPONSE_OK, NULL);
	last_open_dir = cssed_window_get_last_open_dir( window );
	if( last_open_dir != NULL ){
		fullpath = g_strconcat(last_open_dir, G_DIR_SEPARATOR_S, NULL);
		gtk_file_chooser_set_filename( GTK_FILE_CHOOSER (opensel),	fullpath );
		g_free(last_open_dir);
		g_free(fullpath);
	}
	if( gtk_dialog_run(GTK_DIALOG(opensel)) == GTK_RESPONSE_OK ){
		filename = gtk_file_chooser_get_filename( GTK_FILE_CHOOSER(opensel) );
		gtk_widget_destroy(opensel);
		create_and_attach_new_named_doc(window, filename);
		g_free(filename);
	}else{
		gtk_widget_destroy(opensel);
	}
#endif
}

// will add text to the current document at the current document's position
void 
cssed_plugin_add_text_to_document( CssedPlugin* plugin, gchar* text )
{
	CssedWindow* window;
	CssedDoc* doc;

	g_return_if_fail( plugin != NULL );
	
	window = PRIV(plugin)->window;	
	doc = document_get_current( window );
	
	document_add_text( doc, text );	
}

gchar*
cssed_plugin_get_text( CssedPlugin* plugin )
{
	CssedWindow* window;
	CssedDoc* doc;	
	gchar* buffer;
	gint len;
	
	g_return_val_if_fail( plugin != NULL, NULL );
	
	window = PRIV(plugin)->window;	
	doc = document_get_current( window );		
	len = document_get_length( doc );
	buffer = g_malloc0(++len);
	document_get_text(doc, document_get_length(doc), buffer);

	return buffer;	
}
	

void
cssed_plugin_select_text_range( CssedPlugin* plugin, gint start, gint end )
{
	CssedWindow* window;
	CssedDoc* doc;	
	
	g_return_if_fail( plugin != NULL );
	
	window = PRIV(plugin)->window;	
	doc = document_get_current( window );	

	document_set_selection_range( doc, start, end );
}
	
gchar*
cssed_plugin_get_selected_text(  CssedPlugin* plugin  )
{
	CssedWindow* window;
	CssedDoc* doc;	
	gchar* buffer;
	gint start;
	gint end;
	gint len;
	
	g_return_val_if_fail( plugin != NULL, NULL );
	
	window = PRIV(plugin)->window;	
	doc = document_get_current( window );
	start = document_get_selection_start( doc );
	end = document_get_selection_end( doc );
	len = end - start;
	
	if( len <= 0 ) return NULL;
	
	buffer = g_malloc0(++len);
	document_get_selected_text( doc, buffer );

	return buffer;
}
	
void
cssed_plugin_clear_selected_text( CssedPlugin* plugin )
{
	CssedWindow* window;
	CssedDoc* doc;	
	
	g_return_if_fail( plugin != NULL );
	
	window = PRIV(plugin)->window;	
	doc = document_get_current( window );
	document_clear_sel( doc );
}


// get character position at line start
gint
cssed_plugin_get_position_at_line ( CssedPlugin* plugin, gint line )
{
	CssedWindow* window;
	CssedDoc* doc;

	if( plugin == NULL )
		return -1;	
	
	window = PRIV(plugin)->window;	
	doc = document_get_current( window );	
	
	return document_get_position_from_line( doc, line);
}
// undo redo functions, useful to revert to previous state
void
cssed_plugin_undo_document_action ( CssedPlugin* plugin )
{
	CssedWindow* window;
	CssedDoc* doc;

	g_return_if_fail( plugin != NULL );
	
	window = PRIV(plugin)->window;	
	doc = document_get_current( window );	
	document_undo( doc );	
}
	
void
cssed_plugin_redo_document_action ( CssedPlugin* plugin )
{
	CssedWindow* window;
	CssedDoc* doc;

	g_return_if_fail( plugin != NULL );
	
	window = PRIV(plugin)->window;	
	doc = document_get_current( window );		
	document_redo( doc );
}
// search functions
void
cssed_plugin_search_next( CssedPlugin* plugin, gchar* text_to_search )
{
	CssedWindow* window;
	CssedDoc* doc;

	g_return_if_fail( plugin != NULL );
	window = PRIV(plugin)->window;	
	doc = document_get_current( window );		
	document_search_next( doc, text_to_search, 4 ); // WHOLEWORD only
}

void
cssed_plugin_search_prev( CssedPlugin* plugin, gchar* text_to_search )
{
	CssedWindow* window;
	CssedDoc* doc;

	g_return_if_fail( plugin != NULL );
	window = PRIV(plugin)->window;	
	doc = document_get_current( window );		
	document_search_prev( doc, text_to_search, 4 ); // WHOLEWORD only	
}
// save
void
cssed_plugin_save_document( CssedPlugin* plugin )
{
	CssedWindow* window;
	CssedDoc* doc;

	g_return_if_fail( plugin != NULL );
	window = PRIV(plugin)->window;	
	doc = document_get_current( window );	
	document_save( doc );	
}
// css dialogs
void 
cssed_plugin_css_dialog_insert( CssedPlugin* plugin, gchar* key, gpointer class )
{
	CssedWindow* window;
	g_return_if_fail( plugin != NULL );	
	window = PRIV(plugin)->window;	
	cssed_window_css_dialog_insert( window, key, class );	
}

void 
cssed_plugin_css_dialog_remove_by_keyword( CssedPlugin* plugin, gchar* key )
{
	CssedWindow* window;
	g_return_if_fail( plugin != NULL );	
	window = PRIV(plugin)->window;	
	cssed_window_css_dialog_remove_by_keyword( window, key );	
}

gpointer 
cssed_plugin_css_dialog_lookup_by_keyword( CssedPlugin* plugin, gchar* key )
{
	CssedWindow* window;
	if( plugin == NULL ) return NULL;	
	window = PRIV(plugin)->window;	
	return cssed_window_css_dialog_lookup_by_keyword( window, key );	
}

gboolean 
cssed_plugin_css_dialog_keyword_in_use( CssedPlugin* plugin, gchar* key )
{
	CssedWindow* window;
	if( plugin == NULL ) return FALSE;	
	window = PRIV(plugin)->window;	
	return cssed_window_css_dialog_keyword_in_use( window, key );	
}

gpointer
cssed_plugin_get_window( CssedPlugin* plugin )
{
	return (gpointer) PRIV(plugin)->window;
}


