(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

(** The mecanism to handle changes in gui elements. *)

let print_DEBUG s = print_string s ; print_newline ()

open Zog_types

type wid = 
  | WCustom_box of GBin.frame
  | WButton of GButton.button
  | WToggle_button of GButton.toggle_button
  | WCheck_button of GButton.toggle_button
  | WRadio_button of GButton.radio_button
  | WToolbar of GButton.toolbar
  | WHbox of GPack.box
  | WVbox of GPack.box
  | WHbutton_box of GPack.button_box
  | WVbutton_box of GPack.button_box
  | WFixed of GPack.fixed
  | WFrame of GBin.frame
  | WAspect_frame of GBin.aspect_frame
  | WScrolled_window of GBin.scrolled_window
  | WEvent_box of GBin.event_box
  | WHandle_box of GBin.handle_box
  | WViewport of GBin.viewport
  | WHseparator of GObj.widget_full
  | WVseparator of GObj.widget_full
  | WClist of string GList.clist
  | WLabel of GMisc.label
  | WStatusbar of GMisc.statusbar
  | WNotebook of GPack.notebook
  | WColor_selection of GMisc.color_selection
  | WPixmap of GMisc.pixmap 
  | WEntry of GEdit.entry
  | WSpin_button of GEdit.spin_button
  | WCombo of GEdit.combo 
  | WTree of GTree.tree
  | WText of GEdit.text
  | WHpaned of GPack.paned
  | WVpaned of GPack.paned
  | WWindow of GWindow.window
  | WMenubar of GMenu.menu_shell
  | WTable of GPack.table
  | WProgress_bar of GRange.progress_bar
  | WRuler of GRange.ruler
  | WArrow of GMisc.arrow
  | WCalendar of GMisc.calendar
  | WDrawing_area of GMisc.drawing_area
  | WFont_selection of GMisc.font_selection
  | WMenu of GMenu.menu
  | WMenu_item of GMenu.menu_item
  | WTearoff_menu_item of GMenu.menu_item
  | WCheck_menu_item of GMenu.check_menu_item
  | WRadio_menu_item of GMenu.radio_menu_item
  | WMenu_separator of GMenu.menu_item

let create_ele_widget ele =
  match ele.classe with
  | Custom_box -> 
      let w = GBin.frame ~label: "custom" () in
      (WCustom_box w, w#coerce, w#misc)
  | Button ->
      let w = GButton.button () in
      (WButton w, w#coerce, w#misc)
  | Toggle_button ->
      let w = GButton.toggle_button () in
      (WToggle_button w, w#coerce, w#misc)
  | Check_button ->
      let w = GButton.check_button () in
      (WCheck_button w, w#coerce, w#misc)
  | Radio_button ->
      let w = GButton.radio_button () in
      (WRadio_button w, w#coerce, w#misc)
  | Toolbar ->
      let w = GButton.toolbar () in
      (WToolbar w, w#coerce, w#misc)
  | Hbox ->
      let w = GPack.hbox () in
      (WHbox w, w#coerce, w#misc)
  | Vbox ->
      let w = GPack.vbox () in
      (WVbox w, w#coerce, w#misc)
  | Hbutton_box ->
      let w = GPack.button_box `HORIZONTAL () in
      (WHbutton_box w, w#coerce, w#misc)
  | Vbutton_box ->
      let w = GPack.button_box `VERTICAL () in
      (WVbutton_box w, w#coerce, w#misc)
  | Fixed ->
      let w = GPack.fixed () in
      (WFixed w, w#coerce, w#misc)
  | Frame ->
      let w = GBin.frame () in
      (WFrame w, w#coerce, w#misc)
  | Aspect_frame ->
      let w = GBin.aspect_frame () in
      (WAspect_frame w, w#coerce, w#misc)
  | Scrolled_window ->
      let w = GBin.scrolled_window () in
      (WScrolled_window w, w#coerce, w#misc)
  | Event_box ->
      let w = GBin.event_box () in
      (WEvent_box w, w#coerce, w#misc)
  | Handle_box ->
      let w = GBin.handle_box () in
      (WHandle_box w, w#coerce, w#misc)
  | Viewport ->
      let w = GBin.viewport () in
      (WViewport w, w#coerce, w#misc)
  | Hseparator ->
      let w = GMisc.separator `HORIZONTAL () in
      (WHseparator w, w#coerce, w#misc)
  | Vseparator ->
      let w = GMisc.separator `VERTICAL () in
      (WVseparator w, w#coerce, w#misc)
  | Clist ->
      let w = GList.clist () in
      (WClist w, w#coerce, w#misc)
  | Label ->
      let w = GMisc.label () in
      (WLabel w, w#coerce, w#misc)
  | Statusbar ->
      let w = GMisc.statusbar () in
      (WStatusbar w, w#coerce, w#misc)
  | Notebook ->
      let w = GPack.notebook () in
      (WNotebook w, w#coerce, w#misc)
  | Color_selection ->
      let w = GMisc.color_selection () in
      (WColor_selection w, w#coerce, w#misc)
  | Pixmap_file | Pixmap_data | Pixmap_code | Pixmap ->
      (* A VOIR *)
      let w = GBin.frame ~label: "pixmap" () in
      (WFrame w, w#coerce, w#misc)
  | Entry ->
      let w = GEdit.entry () in
      (WEntry w, w#coerce, w#misc)
  | Spin_button ->
      let w = GEdit.spin_button () in
      (WSpin_button w, w#coerce, w#misc)
  | Combo ->
      let w = GEdit.combo () in
      (WCombo w, w#coerce, w#misc)
  | Tree ->
      let w = GTree.tree () in
      (WTree w, w#coerce, w#misc)
  | Text ->
      let w = GEdit.text () in
      (WText w, w#coerce, w#misc)
  | Hpaned ->
      let w = GPack.paned `HORIZONTAL () in
      (WHpaned w, w#coerce, w#misc)
  | Vpaned ->
      let w = GPack.paned `VERTICAL () in
      (WVpaned w, w#coerce, w#misc)
  | Window ->
      let w = GWindow.window () in
      (WWindow w, w#coerce, w#misc)
  | Menubar ->
      let w = GMenu.menu_bar () in
      (WMenubar w, w#coerce, w#misc)

  | Menu_item ->
      let label = Zog_types.get_prop_value ele.props PLabel in
      let item = GMenu.menu_item ~label: label () in
      (WMenu_item item, item#coerce, item#misc)
  | Check_menu_item ->
      let label = Zog_types.get_prop_value ele.props PLabel in
      let item = GMenu.check_menu_item ~label: label () in
      (WCheck_menu_item item, item#coerce, item#misc)
  | Radio_menu_item ->
      let label = Zog_types.get_prop_value ele.props PLabel in
      let item = GMenu.radio_menu_item ~label: label () in
      (WRadio_menu_item item, item#coerce, item#misc)

  | Menu_separator ->
      let item = GMenu.menu_item () in
      (WMenu_separator item, item#coerce, item#misc)

  | Tearoff_menu_item ->
      let item = GMenu.tearoff_item () in
      (WTearoff_menu_item item, item#coerce, item#misc)

  | Menu ->
      let m = GMenu.menu () in
      (WMenu m, m#coerce, m#misc)

  | Table ->
      let t = GPack.table ~rows: 1 ~columns:1 () in
      (WTable t, t#coerce, t#misc)
  | Progress_bar ->
      let p = GRange.progress_bar () in
      (WProgress_bar p, p#coerce, p#misc)
  | HRuler ->
      let r = GRange.ruler `HORIZONTAL () in
      (WRuler r, r#coerce, r#misc)
  | VRuler ->
      let r = GRange.ruler `VERTICAL () in
      (WRuler r, r#coerce, r#misc)
  | Arrow ->
      let a = GMisc.arrow ~kind: `UP ~shadow: `NONE () in
      (WArrow a, a#coerce, a#misc)
  | Calendar ->
      let c = GMisc.calendar () in
      (WCalendar c, c#coerce, c#misc)
  | Drawing_area ->
      let d = GMisc.drawing_area () in
      (WDrawing_area d, d#coerce, d#misc)
  | Font_selection ->
      let w = GMisc.font_selection () in
      (WFont_selection w, w#coerce, w#misc)

let pos_in_parent p_ele ele = 
  let rec iter n = function
      [] -> 0
    | e :: q -> 
	if e == ele then n else iter (n+1) q
  in
  iter 0 p_ele.children

let get_nth_tree_item n wt =
  try Some (List.nth wt#children n)
  with Failure _ -> None

let pack_in_parent w_parent pos widget generic_w ele =
  print_DEBUG ("pack_in_parent "^ele.name) ;
  match w_parent with
  | WCustom_box _ -> ()
  | WButton b -> b#add widget
  | WToggle_button b -> b#add widget
  | WCheck_button b -> b#add widget
  | WRadio_button b -> b#add widget
  | WToolbar tb -> ()
  | WHbox box -> box#pack widget
  | WVbox box -> box#pack widget
  | WHbutton_box box -> box#pack widget
  | WVbutton_box box -> box#pack widget
  | WFixed f -> f#add widget (* A VOIR *)
  | WFrame f -> f#add widget
  | WAspect_frame f -> f#add widget
  | WScrolled_window sw -> 
      (match ele.classe with
	Tree -> sw#add_with_viewport widget
      |	_ -> sw#add widget
      )
  | WEvent_box eb -> eb#add widget
  | WHandle_box hb -> hb#add widget
  | WViewport vp -> vp#add widget
  | WHseparator sep -> ()
  | WVseparator sep -> ()
  | WClist l -> ()
  | WLabel l -> ()
  | WStatusbar sb -> ()
  | WNotebook nb -> 
      let wl = GMisc.label ~text: (Zog_types.get_prop_value ele.props Tab_label) () in
      nb#insert_page ~tab_label: wl#coerce ~pos: pos widget
  | WColor_selection cs -> ()
  | WPixmap p -> ()
  | WEntry e -> ()
  | WSpin_button sb -> ()
  | WCombo c -> ()
  | WTree t -> ()
  | WText t -> ()
  | WHpaned p
  | WVpaned p ->
      (match pos with
	0 -> p#add1 widget
      |	1 -> p#add2 widget
      |	_ -> ()
      )
  | WWindow w ->
      w#add widget
  | WTable t -> ()
  | WProgress_bar p -> ()
  | WRuler r -> ()
  | WArrow a -> ()
  | WCalendar c -> ()
  | WDrawing_area d -> ()
  | WFont_selection cs -> ()

  | WMenu m ->
      (
       match generic_w with
	 WMenu_item i -> m#insert ~pos: pos i
       | WCheck_menu_item i -> m#insert ~pos: pos (i :> GMenu.menu_item)
       | WRadio_menu_item i -> m#insert ~pos: pos (i :> GMenu.menu_item)
       | WMenu_separator i -> m#insert ~pos: pos i
       | WTearoff_menu_item i -> m#insert ~pos: pos i
       | _ ->
	  () 
      )
  | WMenubar m ->
      (
       print_DEBUG "pack menubar" ;
       match generic_w with
	 WMenu_item i -> 
	   print_DEBUG "packing menu_item";
	   m#insert ~pos: pos i 
       | WCheck_menu_item i -> m#insert ~pos: pos (i :> GMenu.menu_item)
       | WRadio_menu_item i -> m#insert ~pos: pos (i :> GMenu.menu_item)
       | WMenu_separator i -> m#insert ~pos: pos i
       | WTearoff_menu_item i -> m#insert ~pos: pos i	     
       | _ ->
	   print_DEBUG "_ case";
	  () 
      )
  | WMenu_item i ->
      (
       match generic_w with
	 WMenu m -> i#set_submenu m
       | _ -> ()
      )
  | WCheck_menu_item i -> ()
  | WRadio_menu_item i -> ()
  | WMenu_separator i -> ()
  | WTearoff_menu_item i -> ()

let unpack_from_parent w_parent pos widget generic_w ele =
  match w_parent with
  | WCustom_box _ -> ()
  | WButton b -> b#remove widget
  | WToggle_button b -> b#remove widget
  | WCheck_button b -> b#remove widget
  | WRadio_button b -> b#remove widget
  | WToolbar tb -> ()
  | WHbox box -> box#remove widget
  | WVbox box -> box#remove widget
  | WHbutton_box box -> box#remove widget
  | WVbutton_box box -> box#remove widget
  | WFixed f -> f#remove widget (* A VOIR *)
  | WFrame f -> f#remove widget
  | WAspect_frame f -> f#remove widget
  | WScrolled_window sw -> 
      (match ele.classe with
	Tree -> 
	  (match sw#children with
	    ch :: _ -> sw#remove ch
	  | _ -> ()
	  )
      |	_ -> sw#remove widget
      )
  | WEvent_box eb -> eb#remove widget
  | WHandle_box hb -> hb#remove widget
  | WViewport vp -> vp#remove widget
  | WHseparator sep -> ()
  | WVseparator sep -> ()
  | WClist l -> ()
  | WLabel l -> ()
  | WStatusbar sb -> ()
  | WNotebook nb -> 
      (
       try nb#remove_page (nb#page_num widget)
       with _ -> ()
      )
  | WColor_selection cs -> ()
  | WPixmap p -> ()
  | WEntry e -> ()
  | WSpin_button sb -> ()
  | WCombo c -> ()
  | WTree t -> ()
  | WText t -> ()
  | WHpaned p
  | WVpaned p ->
      (match pos with
	0 -> p#remove widget
      |	1 -> p#remove widget
      |	_ -> ()
      )
  | WWindow w ->
      w#remove widget
  | WTable t -> ()
  | WProgress_bar p -> ()
  | WRuler r -> ()
  | WArrow a -> ()
  | WCalendar c -> ()
  | WDrawing_area d -> ()
  | WFont_selection cs -> ()

  | WMenu m ->
      (
       match generic_w with
	 WMenu_item i -> m#remove i
       | WCheck_menu_item i -> m#remove (i :> GMenu.menu_item)
       | WRadio_menu_item i -> m#remove (i :> GMenu.menu_item)
       | WMenu_separator i -> m#remove i
       | WTearoff_menu_item i -> m#remove i
       | _ ->
	  () 
      )
  | WMenubar m ->
      (
       match generic_w with
	 WMenu_item i -> m#remove i
       | WCheck_menu_item i -> m#remove (i :> GMenu.menu_item)
       | WRadio_menu_item i -> m#remove (i :> GMenu.menu_item)
       | WMenu_separator i -> m#remove i
       | WTearoff_menu_item i -> m#remove i
       | _ ->
	  () 
      )
  | WMenu_item i ->
      (
       match generic_w with
	 WMenu m -> i#remove_submenu ()
       | _ -> ()
      )
  | WCheck_menu_item i -> ()
  | WRadio_menu_item i -> ()
  | WMenu_separator i -> ()
  | WTearoff_menu_item i -> ()

let set_border_width w n =
  match w with
  | WCustom_box _ -> ()
  | WButton b -> b#set_border_width n
  | WToggle_button b -> b#set_border_width n
  | WCheck_button b -> b#set_border_width n
  | WRadio_button b -> b#set_border_width n
  | WToolbar tb -> ()
  | WHbox box -> box#set_border_width n
  | WVbox box -> box#set_border_width n
  | WHbutton_box box -> box#set_border_width n
  | WVbutton_box box -> box#set_border_width n
  | WFixed f -> f#set_border_width n
  | WFrame f -> f#set_border_width n
  | WAspect_frame f -> f#set_border_width n
  | WScrolled_window sw -> sw#set_border_width n
  | WEvent_box eb -> eb#set_border_width n
  | WHandle_box hb -> hb#set_border_width n
  | WViewport vp -> vp#set_border_width n
  | WHseparator sep -> ()
  | WVseparator sep -> ()
  | WClist l -> l#set_border_width n
  | WLabel l -> ()
  | WStatusbar sb -> sb#set_border_width n
  | WNotebook nb -> nb#set_border_width n
  | WColor_selection cs -> ()
  | WPixmap p -> ()
  | WEntry e -> ()
  | WSpin_button sb -> ()
  | WCombo c -> ()
  | WTree t -> t#set_border_width n
  | WText t -> ()
  | WHpaned p
  | WVpaned p ->p#set_border_width n
  | WWindow w ->
      w#set_border_width n
  | WMenubar m -> m#set_border_width n
  | WTable t -> t#set_border_width n
  | WProgress_bar p -> ()
  | WRuler r -> ()
  | WArrow a -> ()
  | WCalendar c -> ()
  | WDrawing_area d -> ()
  | WFont_selection cs ->  ()
  | WMenu _ -> ()
  | WMenu_item i -> ()
  | WCheck_menu_item i -> ()
  | WRadio_menu_item i -> ()
  | WMenu_separator i -> ()
  | WTearoff_menu_item i -> ()

let enum_v ele l prop_kind = 
  try 
    let v = Zog_types.get_prop_value ele.props prop_kind in
    List.assoc v l
  with
    Not_found -> snd (List.hd l)

let bool_v ele prop_kind =
  try
    let v = Zog_types.get_prop_value ele.props prop_kind in
    List.assoc v Zog_types.bool_values
  with
    Not_found ->
      match prop_kind with
	Expand | Homogeneous | Right_justify -> false
      |	_ -> true

let change_packing_properties parent_w widget ele =
  let f 
      (f_set: ?from:Gtk.Tags.pack_type -> 
	?expand:bool -> ?fill:bool -> ?padding:int -> GObj.widget -> unit) =
    let b_expand = bool_v ele Expand in 
    let b_fill = bool_v ele Fill in
    try 
      let n = int_of_string (Zog_types.get_prop_value ele.props Padding) in
      if n < 0 then
	raise (Failure "")
      else
	f_set ~expand: b_expand ~fill:b_fill ~padding: n widget
    with
      Invalid_argument _ | Failure _ ->
	f_set ~expand: b_expand ~fill:b_fill widget
  in
  let f_set =
    match parent_w with
    | WHbox box -> box#set_child_packing
    | WVbox box -> box#set_child_packing
    | WHbutton_box box -> box#set_child_packing
    | WVbutton_box box -> box#set_child_packing
    | WCustom_box _ | WButton _ | WToggle_button _ | WCheck_button  _
    | WRadio_button _ | WToolbar _ | WFixed _ | WFrame _ | WAspect_frame _
    | WScrolled_window _ | WEvent_box _ | WHandle_box _ | WViewport _
    | WHseparator _ | WVseparator _ | WClist _ | WLabel _ | WStatusbar _
    | WNotebook _ | WColor_selection _ | WPixmap _ | WEntry _
    | WSpin_button _ | WCombo _ | WTree _ | WText _ | WHpaned _
    | WVpaned _ | WWindow _ | WMenubar _ | WTable _ | WProgress_bar _ 
    | WRuler _ | WArrow _ | WCalendar _ | WDrawing_area _ 
    | WFont_selection _ | WMenu _ | WMenu_item _ | WRadio_menu_item _ 
    | WCheck_menu_item _ | WMenu_separator _ | WTearoff_menu_item _ ->
	let g ?from ?expand ?fill ?padding w = () in g
  in
  f f_set
    
let get_int_prop_value ele prop_kind =
  let n_str = Zog_types.get_prop_value ele.props prop_kind in
  try
    let n = int_of_string n_str in
    if n >= 0 then Some n else None
  with
     _ -> None

let get_float_prop_value ele prop_kind =
  let f_str = Zog_types.get_prop_value ele.props prop_kind in
  try
    let f = float_of_string f_str in
    if f >= 0. then Some f else None
  with
     _ -> None

let map_opt f o =
  match o with
    None -> None
  | Some x -> Some (f x)

let change_geometry_properties misc ele =
  misc#set_geometry
    ?x: (get_int_prop_value ele X_pos)
    ?y: (get_int_prop_value ele Y_pos)
    ?width: (get_int_prop_value ele Width)
    ?height: (get_int_prop_value ele Height)
    ()

class handler =
  object (self)
    val mutable eles = 
      ([] : (gui_element * gui_element option * 
	       GMisc.label * wid * GObj.widget * GObj.misc_ops) list)

    (** There can be only one window handled by the handler. *)
    val mutable window = (None : GWindow.window option)

    val mutable last_enter_ele = None

    method window = window

    method add_ele parent_opt e label =
      print_DEBUG ("handler#add_ele: "^e.name);
      self#set_label_style e label ;
      try
	let _ = self#get_info e in
	print_DEBUG ("handler#add_ele: info on "^e.name^" not found");
	()
      with
	Not_found ->
	  let (w, widget, misc) = create_ele_widget e in
	  eles <- (e, parent_opt, label, w, widget, misc) :: eles ;
(*
	  (**/**)(*TEST*)
	  let ev = new GObj.event_ops (GObj.as_widget widget) in
	  ev#connect#enter_notify
	    (fun _ -> 
	      match last_enter_ele with
		None ->
	      	  prerr_endline ("enter notify on "^e.name);
		  last_enter_ele <- Some e ;
		  true
	      | Some e2 ->
		  if e2 != e then
		    (prerr_endline ("enter notify on "^e.name);
		     last_enter_ele <- Some e 
		    )
		  else
		    ();
		  true
	    );
	  (**/**)(*/TEST*)
*)

	  (match parent_opt, e.classe with
	    None, Window ->
	      (
	       match w with
		 WWindow wdw -> 
		   let _ = wdw#event#connect#delete (fun _ -> true) in
		   window <- Some wdw
	       | _ -> ()
	      ) ;
	      misc#show ()
	  | None, _ ->
	      let dummy_w = GWindow.window ~border_width: 20 () in
	      let w = dummy_w#misc in
	      let style = w#style#copy in
              w#set_style style;
              style#set_bg [ (`NORMAL, `NAME "SkyBlue") ;
                             (`INSENSITIVE, `NAME "SkyBlue") ;
                             (`PRELIGHT, `NAME "SkyBlue") ;
                             (`ACTIVE, `NAME "SkyBlue") ;
                             (`SELECTED, `NAME "SkyBlue")
			   ] ;

	      window <- Some dummy_w ;
	      dummy_w#add widget ;
	      let _ = dummy_w#event#connect#delete (fun _ -> true) in
	      dummy_w#show ()
	  | Some p_ele, Window ->
  	      (* should not occur *)
	      misc#show ()
	  | Some p_ele, _ ->
	      try
		let (_,_,_,w_parent,_,_) = self#get_info p_ele in
		let pos = pos_in_parent p_ele e in
		pack_in_parent w_parent pos widget w e
	      with
		Not_found ->
		  ()
	  );
	  List.iter (self#property_changed e) e.props

    method remove_ele ele =
      let rec children e =
	e :: 
	(List.flatten
	   (List.map children e.children))
      in
      let to_be_removed = children ele in
      let has_parent = 
	try 
	  let (_, parent_opt, _, _, _, _) = self#get_info ele in
	  parent_opt <> None
	with
	  Not_found -> true 
      in

      List.iter
	(fun (e,parent_opt,_,w,widget,misc) -> 
	  if e == ele then
	    (
	     List.iter self#remove_ele e.children ;
	     match parent_opt with
	       None -> misc#hide ()
	     | Some p_ele ->
		 try
		   let (_,_,_,w_parent,_,_) = self#get_info p_ele in
		   let pos = pos_in_parent p_ele e in
		   unpack_from_parent w_parent pos widget w e
		 with
		   Not_found -> ()
	    )
	) 
	eles ;
      eles <- List.filter 
	  (fun (e,_,_,_,_,_) -> not (List.memq e to_be_removed)) eles ;

      (* destroy the window if it is a window or if it was 
	 the root of a box without window. *)
      match ele.classe, window, has_parent with
	Window, Some w, _ -> w#destroy () ; window <- None
      |	_, _, false -> 
	  (
	   match window with
	     Some w -> w#destroy () ; window <- None
	   | None -> ()
	  )
      |	_ -> ()

    method up_element parent_ele ele =
      (* The children in the parent_ele are in the correct order. *)
      try
	let (_,_,label,_,widget,_) = self#get_info ele in
	let (_,_,_,w,_,_) = self#get_info parent_ele in
	let pos = pos_in_parent parent_ele ele in
	match w with
	| WCustom_box _ -> ()
	| WButton _ | WToggle_button _ | WCheck_button _
	| WRadio_button _ -> ()
	      
	| WToolbar tb -> (* A VOIR quand on grera les boutons dedans *)
	    ()

	| WHbox box
	| WVbox box -> box#reorder_child widget ~pos: pos
	| WHbutton_box box
	| WVbutton_box box -> box#reorder_child widget ~pos: pos
	| WFixed f -> () (* A VOIR *)
	      
	| WFrame _ | WAspect_frame _ | WScrolled_window _
	| WEvent_box _ | WHandle_box _ | WViewport _
	| WHseparator _ | WVseparator _ | WClist _
	| WLabel _ | WStatusbar _ 
	| WColor_selection _ | WPixmap _ | WEntry _
	| WSpin_button _ | WCombo _ | WTree _
	| WText _ | WTable _ 
	| WProgress_bar _ | WRuler _ | WArrow _
	| WCalendar _ | WDrawing_area _ 
	| WFont_selection _ ->
	    ()
	      
	| WHpaned p
	| WVpaned p ->
 	    (* remove and add each child *)
	    let liste = List.fold_left
		(fun acc -> fun e ->
		  try
		    let (_,_,label,_,_,_) = self#get_info e in
		    acc @ [e, label]
		  with
		    Not_found -> acc
		)
		[]
		parent_ele.children
	    in
	    List.iter (fun (e, l) -> self#remove_ele e) liste ;
	    List.iter (fun (e, l) -> self#add_ele (Some parent_ele) e l) liste

	| WNotebook _ ->
 	    (* remove and add each child *)
	    List.iter
	      (fun e ->
		try 
		  let (_,_,label,_,_,_) = self#get_info e in
		  self#remove_ele e ;
		  self#add_ele (Some parent_ele) e label
		with
		  Not_found ->
		    ()
	      )
	      parent_ele.children

	| WWindow w ->
	    ()

	| WMenubar mb ->
	    (* remove and add each child *)
	    List.iter
	      (fun e ->
		try 
		  let (_,_,label,_,_,_) = self#get_info e in
		  self#remove_ele e ;
		  self#add_ele (Some parent_ele) e label
		with
		  Not_found ->
		    ()
	      )
	      parent_ele.children

	| WMenu m ->
	    (* remove and add each child *)
	    List.iter
	      (fun e ->
		try 
		  let (_,_,label,_,_,_) = self#get_info e in
		  self#remove_ele e ;
		  self#add_ele (Some parent_ele) e label
		with
		  Not_found ->
		    ()
	      )
	      parent_ele.children

	| WMenu_item _ 
	| WCheck_menu_item _ 
	| WRadio_menu_item _
	| WMenu_separator _ 
	| WTearoff_menu_item _ ->
	    ()
      with
	Not_found -> ()

    method private get_info ele =
      List.find (fun (e,_,_,_,_,_) -> e == ele) eles

    method set_label_style ele label =
      if Zog_data.in_interface ele then 
	(
	 let font = Gdk.Font.load "*bold*" in
	 let style = label#misc#style#copy in
	 label#misc#set_style style;
	 style#set_font font
	)
      else
	let l = GMisc.label () in
	label#misc#set_style l#misc#style#copy
      
    method ele_name_changed ele =
      try
	let (_,_,label,_,_,_) = self#get_info ele in
	label#set_text ele.name;
	self#set_label_style ele label
      with
	Not_found -> ()

    method property_changed (ele : gui_element) (p : property) =
      try 
	let (e,parent_opt,_,w,widget,misc) = self#get_info ele in
	match p.prop_kind with
	| Function -> ()
	| Fill | Padding | Expand ->
	    (
	     match parent_opt with
	       None -> ()
	     | Some p_ele ->
		 try
		   let (_,_,_,parent_w,_,_) = self#get_info p_ele in
		   change_packing_properties parent_w widget ele
		 with Not_found -> ()
	    )
	| X_pos | Y_pos | Height | Width ->
	    change_geometry_properties misc ele
	| Border_width ->
	    Zog_misc.apply_opt (set_border_width w) (get_int_prop_value ele Border_width)
	| Title ->
	    (match w with
	       WWindow window -> 
		 window#set_title (Zog_types.get_prop_value ele.props Title)
	     | _ -> ()
	    )
	| Allow_shrink ->
	    (match w with
	      WWindow window ->	window#set_allow_shrink (bool_v ele Allow_shrink)
	     | _ -> ()
	    )
	| Allow_grow ->
	    (match w with
	      WWindow window ->	window#set_allow_grow (bool_v ele Allow_grow)
	     | _ -> ()
	    )
	| Auto_shrink ->
	    (match w with
	      WWindow window ->	window#set_auto_shrink (bool_v ele Auto_shrink)
	     | _ -> ()
	    )
	| PLabel ->
	    let s = Zog_types.get_prop_value ele.props PLabel in
	    let set_button_label b =
	      match b#children with
	      |	[l] -> (GMisc.label_cast l)#set_text s
	      |	_ -> ()
	    in
	    (match w with
	    | WFrame f -> f#set_label s
	    | WAspect_frame f -> f#set_label s
	    | WButton b -> set_button_label b
	    | WToggle_button b 
	    | WCheck_button b -> set_button_label b 
	    | WRadio_button b -> set_button_label b
	    | _ -> ()
	    )
	| Group ->
	    (* nothing to do *)
	    ()
	| Orientation ->
	    (match w with
	      WToolbar t ->
		let o = enum_v ele Zog_types.orientation_values Orientation in
		t#set_orientation o
	    | _ -> ()
	    )
	| Toolbar_style ->
	    (match w with
	      WToolbar t ->
		let s = enum_v ele Zog_types.toolbar_style_values Toolbar_style in
		t#set_style s
	    | _ -> ()
	    )
	| Toolbar_space_size ->
	    (match w with
	      WToolbar t -> 
		Zog_misc.apply_opt t#set_space_size (get_int_prop_value ele Toolbar_space_size)
	    | _ -> ()
	    )
	| Toolbar_space_style ->
	    (match w with
	      WToolbar t -> 
		let s = enum_v ele Zog_types.toolbar_space_values Toolbar_space_style in
		t#set_space_style s
	    | _ -> ()
	    )
	| Tooltips ->
	    (match w with
	      WToolbar t -> t#set_tooltips (bool_v ele Tooltips)
	    | _ -> ()
	    )
	| Button_relief_style ->
	    (match w with
	      WToolbar t -> 
		let s = enum_v ele Zog_types.button_relief_values Button_relief_style in
		t#set_button_relief s
	    | _ -> ()
	    )
	| Spacing ->
	    Zog_misc.apply_opt
	      (match w with
		WHbox b | WVbox b -> b#set_spacing
	      |	WHbutton_box b | WVbutton_box b -> b#set_spacing
	      |	_ -> fun x -> ())
	      (get_int_prop_value ele Spacing)
	| Homogeneous ->
	    (match w with
	      WHbox b | WVbox b -> b#set_homogeneous (bool_v ele Homogeneous)
	    | _ -> ()
	    )
	| Button_box_style ->
	    (match w with
	      WHbutton_box b | WVbutton_box b -> 
		b#set_layout 
		  (enum_v ele Zog_types.button_box_style_values	Button_box_style)
	    | _ -> ()
	    )
	| Child_width | Child_height ->
	    (match w with
	      WHbutton_box b 
	    | WVbutton_box b -> 
		b#set_child_size 
		  ?width: (get_int_prop_value ele Child_width)
		  ?height: (get_int_prop_value ele Child_height) ();
                (* need hide/show to update the display of the button_box *)
		b#misc#hide () ; b#misc#show () 
	    | _ -> ()
	    )
	| Child_ipadx | Child_ipady ->
	    (match w with
	      WHbutton_box b 
	    | WVbutton_box b -> 
		b#set_child_ipadding
		  ?x: (get_int_prop_value ele Child_ipadx)
		  ?y: (get_int_prop_value ele Child_ipady) ();
                (* need hide/show to update the display of the button_box *)
		b#misc#hide () ; b#misc#show () 
	    | _ -> ()
	    )
	| Label_xalign | Label_yalign ->
	    (match w with
	      WFrame f ->
		f#set_label_align
		  ?x: (get_float_prop_value ele Label_xalign)
		  ?y: (get_float_prop_value ele Label_yalign) ()
	    | WAspect_frame f -> 
		f#set_label_align
		  ?x: (get_float_prop_value ele Label_xalign)
		  ?y: (get_float_prop_value ele Label_yalign) ()
	    | _ -> ()
	    )
	| Shadow_type ->
	    (match w with
	      WFrame f -> f#set_shadow_type
	    | WAspect_frame f -> f#set_shadow_type
	    | WViewport v -> v#set_shadow_type
	    | WClist l -> l#set_shadow_type
	    | WSpin_button s -> s#set_shadow_type
	    | WArrow a -> 
		let type_v = enum_v ele Zog_types.arrow_type_values Arrow_type in
		fun st -> a#set_arrow type_v ~shadow: st
	    | _ -> fun _ -> ()
	    )
	      (enum_v ele Zog_types.shadow_type_values Shadow_type)
	| Obey_child ->
	    (match w with
	    | WAspect_frame f -> f#set_obey_child (bool_v ele Obey_child)
	    | _ -> ()
	    )
	| Ratio ->
	    (match w with
	    | WAspect_frame f -> 
		Zog_misc.apply_opt f#set_ratio (get_float_prop_value ele Ratio)
	    | _ -> ()
	    )
	| Hscrollbar_policy ->
	    (match w with
	    | WScrolled_window sb -> sb#set_hpolicy
		  (enum_v ele Zog_types.policy_type_values Hscrollbar_policy)
	    | _ -> ()
	    )
	| Vscrollbar_policy ->
	    (match w with
	    | WScrolled_window sb -> sb#set_vpolicy
		  (enum_v ele Zog_types.policy_type_values Vscrollbar_policy)
	    | _ -> ()
	    )
	| Handle_position ->
	    (match w with
	    | WHandle_box b -> b#set_handle_position
		  (enum_v ele Zog_types.position_values Handle_position)
	    | _ -> ()
	    )
	| Snap_edge ->
	    (match w with
	    | WHandle_box b -> b#set_snap_edge
		  (enum_v ele Zog_types.position_values Snap_edge)
	    | _ -> ()
	    )
	| Show_titles ->
	    (match w with
	    | WClist l -> l#set_titles_show (bool_v ele Show_titles)
	    | _ -> ()
	    )
	| X_align | Y_align ->
	    let f = get_float_prop_value ele in
	    let x_opt = f X_align in
	    let y_opt = f Y_align in
	    (match w with
	    | WAspect_frame f -> f#set_alignment ?x: x_opt ?y:y_opt ()
	    | WLabel l -> l#set_alignment ?x: x_opt ?y:y_opt ()
	    | WPixmap p -> p#set_alignment ?x: x_opt ?y:y_opt ()
	    | WArrow a -> a#set_alignment ?x: x_opt ?y:y_opt ()
	    | _ -> ()
	    )
	| X_pad | Y_pad ->
	    let f = get_int_prop_value ele in
	    let x_opt = f X_pad in
	    let y_opt = f Y_pad in
	    (match w with
	    | WLabel l -> l#set_padding ?x: x_opt ?y:y_opt ()
	    | WPixmap p -> p#set_padding ?x: x_opt ?y:y_opt ()
	    | WArrow a -> a#set_padding ?x: x_opt ?y:y_opt ()
	    | _ -> ()
	    )
	| PText ->
	    (match w with
	      WLabel l -> l#set_text (Zog_types.get_prop_value ele.props PText)
	    | WEntry e -> e#set_text (Zog_types.get_prop_value ele.props PText)
	    | _ -> () 
	    )
	| Line_wrap ->
	    (match w with
	    | WLabel l -> l#set_line_wrap (bool_v ele Line_wrap)
	    | WText t -> t#set_line_wrap (bool_v ele Line_wrap)
	    | _ -> ()
	    )
	| Tab_pos ->
	    (match w with
	      WNotebook n -> n#set_tab_pos
		  (enum_v ele Zog_types.position_values Tab_pos)
	    | _ -> ()
	    )
	| Show_tabs ->
	    (match w with
	      WNotebook n -> n#set_show_tabs (bool_v ele Show_tabs)
	    | _ -> ()
	    )
	| Homogeneous_tabs ->
	    (match w with
	      WNotebook n -> n#set_homogeneous_tabs (bool_v ele Homogeneous_tabs)
	    | _ -> ()
	    )
	| Show_border ->
	    (match w with
	      WNotebook n -> n#set_show_border (bool_v ele Show_border)
	    | _ -> ()
	    )
	| Scrollable ->
	    (match w with
	      WNotebook n -> n#set_scrollable (bool_v ele Scrollable)
	    | _ -> ()
	    )
	| Tab_border ->
	    (match w with
	      WNotebook n -> 
		Zog_misc.apply_opt n#set_tab_border (get_int_prop_value ele Tab_border)
	    | _ -> ()
	    )
	| Popup ->
	    (match w with
	      WNotebook n -> n#set_popup (bool_v ele Popup)
	    | _ -> ()
	    )
	| SBUpdate_policy ->
	    (match w with
	      WSpin_button sb -> sb#set_update_policy
		  (enum_v ele Zog_types.sb_update_type_values SBUpdate_policy)
	    | _ -> ()
	    )
	| Visibility ->
	    (match w with
	      WEntry e -> e#set_visibility (bool_v ele Visibility)
	    | _ -> ()
	    )
	| Editable ->
	    (match w with
	    | WEntry e -> e#set_editable (bool_v ele Editable)
	    | WText t -> t#set_editable (bool_v ele Editable)
	    | _ -> ()
	    )
	| Use_arrows ->
	    (match w with
	      WCombo w -> w#set_use_arrows 
		  (enum_v ele Zog_types.use_arrows_values Use_arrows)
	    | _ -> ()
	    )
	| Case_sensitive ->
	    (match w with
	      WCombo w -> w#set_case_sensitive (bool_v ele Case_sensitive)
	    | _ -> ()
	    )
	| Word_wrap ->
	    (match w with
	    | WText t -> t#set_word_wrap (bool_v ele Word_wrap)
	    | _ -> ()
	    )
	| Draw_indicator ->
	    (match w with
	      WCheck_button b
	    | WToggle_button b -> b#set_draw_indicator (bool_v ele Draw_indicator)
	    | WRadio_button b -> b#set_draw_indicator (bool_v ele Draw_indicator)
	    | _ -> ()
	    )
	| Active ->
	     (match w with
	      WCheck_button b
	    | WToggle_button b -> b#set_active (bool_v ele Active)
	    | WRadio_button b -> b#set_active (bool_v ele Active)
	    | _ -> ()
	    )
	| Placement ->
	    (match w with
	      WScrolled_window sc -> sc#set_placement
		  (enum_v ele Zog_types.corner_type_values Placement)
	    | _ -> ()
	    )
	| Selection_mode ->
	    (match w with
	      WClist l -> l#set_selection_mode
		  (enum_v ele Zog_types.selection_mode_values Selection_mode)
	    | WTree t -> t#set_selection_mode
		  (enum_v ele Zog_types.selection_mode_values Selection_mode)
	    | _ -> ()
	    )
	| Justification ->
	    (match w with
	      WLabel l -> l#set_justify 
		  (enum_v ele Zog_types.justification_values Justification)
	    | _ -> ()
	    )
	| Max_length ->
	    (match w with
	      WEntry e -> 
		Zog_misc.apply_opt e#set_max_length (get_int_prop_value ele Max_length)
	    | _ -> ()
	    )
	| View_mode ->
	    (match w with
	    | WTree t -> t#set_view_mode
		  (enum_v ele Zog_types.view_mode_values View_mode)
	    | _ -> ()
	    )
	| View_lines ->
	    (match w with
	    | WTree t -> t#set_view_lines (bool_v ele View_lines)
	    | _ -> ()
	    )
	| Handle_size ->
	    (
	     match w with
	       WHpaned p | WVpaned p -> 
		 Zog_misc.apply_opt p#set_handle_size (get_int_prop_value ele Handle_size)
	     | _ -> ()
	    )
	| Modal ->
	    (* don't do anything, or we could be blocked *)
	    ()
	| Tab_label ->
	    (match parent_opt with
	      None -> ()
	    | Some p_ele ->
		try
		  let (_,_,_,parent_w,_,_) = self#get_info p_ele in
		  match parent_w with
		    WNotebook nb ->
		      let wl = GMisc.label_cast (nb#get_tab_label widget) in
		      wl#set_text (Zog_types.get_prop_value ele.props Tab_label) ;
		  | _ -> ()
		with
		  _ -> ()
	    )

	| Column_number ->
	    (* A VOIR ? *)
	    ()
	| Column_titles ->
	    (match w with
	      WClist w ->
		let s = Zog_types.get_prop_value ele.props Column_titles in
		w#set_column ~title: s 0
	    | _ -> ()
	    )
	| Right_justify ->
	    ( (* A VOIR *)
	     (*if bool_v ele Right_justify then
	       match w with
		 WMenu_item i -> i#right_justify ()
	       | WCheck_menu_item i -> i#right_justify ()
	       | WRadio_menu_item i -> i#right_justify ()
	       | _ -> ()
	     *)
	    )
	| Show_indicator
	| Show_toggle ->
	    (
	     let b_ind = bool_v ele Show_indicator in
	     let b_tog = bool_v ele Show_toggle in
	     match w with
	       | WCheck_menu_item i -> 
		   i#configure ~show_toggle:b_tog ~show_indicator:b_ind 
	       | WRadio_menu_item i -> 
		   i#configure ~show_toggle:b_tog ~show_indicator:b_ind 
	       | _ -> ()
	    )
	| Accel_keysym|Accel_flags| Accel_modifier ->
	    (* A VOIR *)
	    ()
	| Accel_group_name ->
	    ()

	| Arrow_type ->
	    (match w with
	      WArrow a ->
		let type_v = enum_v ele Zog_types.arrow_type_values Arrow_type in
		let shad_t = enum_v ele Zog_types.shadow_type_values Shadow_type in
		a#set_arrow type_v ~shadow: shad_t
	    | _ -> ()
	    )
	| Calendar_options ->
	    (match w with
	      WCalendar c ->
	    	let options = enum_v ele Zog_types.calendar_options_values Calendar_options in
		c#display_options options
	    | _ -> ()
	    )
	| Popdown_strings ->
	    (match w with
	      WCombo c ->
		let s = Zog_types.get_prop_value ele.props Popdown_strings in
		let l =
		  try 
		    let n = String.index s '[' in
		    let s2 = String.sub s (n+1) ((String.length s) - n - 1) in
		    let n2 = String.index s2 ']' in
		    let s3 = String.sub s2 0 n2 in
		    Str.split (Str.regexp_string ";") s3
		  with
		    Not_found -> [s]
		in
		c#set_popdown_strings l
	    | _ -> ()
	    )
	| Value_in_list | Ok_if_empty ->
	    (match w with
	      WCombo c ->
		let b_value = bool_v ele Value_in_list in
		let b_ok = bool_v ele Ok_if_empty in
		c#set_value_in_list ~required: b_value ~ok_if_empty: b_ok ()
	    | _ ->
		()
	    )
	| Update_policy ->
	    (match w with
	      WColor_selection c ->
		c#set_update_policy (enum_v ele Zog_types.update_type_values Update_policy)
	    | _ -> ()
	    )
	| PPixmap_file | PPixmap_data | PPixmap_code ->
	    ()
	with
	    Not_found ->
	      ()

  end

