(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

(** Gui to describe gui entities. *)

let print_DEBUG s = print_string s ; print_newline ()

open Zog_types
open GdkKeysyms

module C = Configwin

let widget_shortcuts_pre = [
  Hbox, [_H] ;
  Vbox, [_h] ;
  Hpaned, [_P] ;
  Vpaned, [_p] ;
  Button, [_B ; _b] ;
  Label, [_L ; _l] ;
  Text, [_T ; _t] ;
  Entry, [_E ; _e] ;
  Menu, [_M ; _m] ;
  Menu_item, [_I ; _i] ;
  Window, [_W ; _w] ;
  Tree, [_T ; _t] ;
] 

(** Shortcuts for inserting widgets in the gui_entity window. *)
let widget_shortcuts =
  List.fold_left
    (fun acc -> fun (cl, keys) ->
      acc @ (List.map (fun k -> (k, cl)) keys))
    []
    widget_shortcuts_pre


(** The selected gui element and its item, with the parent element and its tree. *)
type t_selected_node = 
    gui_element * GTree.tree_item * GMisc.label * 
      gui_element option * GTree.tree

(** The class to build a box for editing a gui entity. *)
class gui_entity data (entity : entity) (f_save : unit -> unit) =
  object (self)
    inherit Zog_gui_base.gui_entity ()

    val handler = new Zog_gui_handler.handler

    val mutable selected_node = (None : t_selected_node option)

    val mutable prop_boxes = ([] : (gui_element * Zog_gui_prop.box) list)
    val mutable widgets_buttons = ([] : (Zog_types.ele_class * GButton.button) list)

    method box = box
    method accel_group = accel_group
    method handler = handler

    method preview = wb_preview#active

(** {2 Copy/cut/paste/delete methods} *)

    method copy () =
      match selected_node with
	None -> ()
      |	Some (ele, _, _, _, _) -> 
	  Zog_buffer.copy ele

    method cut ?(copy=true) () =
      match selected_node with
	None -> ()
      |	Some (ele, item, _, parent_opt, tree) ->
	  selected_node <- None ;
	  if copy then Zog_buffer.copy ele ;
	  prop_boxes <- List.filter 
	      (fun (e, b) -> 
		(e != ele) or 
		(b#box#misc#hide () ; false))
	      prop_boxes ;
	  data#entity_remove_from_parent entity ele parent_opt ;
	  handler#remove_ele ele ;
	  tree#remove_items [item]

    method paste_append () =
      match Zog_buffer.paste () with
	None -> ()
      | Some ele_paste ->
	  let (ele_opt, tree_opt, item_opt) = 
	    match selected_node with
	      None -> (None, Some wtree, None)
	    | Some (ele, item, _, parent_opt, tree) -> 
	        (* when we remove the last item of a tree,
		  this tree gets deassociated from its parent item.
	           We recreate a subtree here, if needed *)
		let sub_opt = 
		  match item#subtree with
		    None ->
		      let wt_sub = GTree.tree () in
		      let _ = item#set_subtree wt_sub in
		      let _ = item#expand in
		      Some (wt_sub)
		  | Some wt -> 
		      print_DEBUG "dont recreate subtree (2)" ;
		      Some wt
		in
		(Some ele, sub_opt, Some item)
	  in
	  match tree_opt with
	    None -> 		
	      print_DEBUG "paste_append : no subtree" ;
	      ()
	  | Some tree ->
	      print_DEBUG "paste_append : some subtree" ;
	      try
		data#entity_append_in_parent entity ele_paste ele_opt ;
		self#insert_gui_ele tree ele_opt ele_paste ;
		match item_opt with
		  None -> ()
		| Some i -> i#expand ()
	      with
		Failure s ->
		  GToolbox.message_box Zog_messages.error s

    method delete () = self#cut ~copy: false ()

(** {2 Selection and deselection of nodes} *)

    method update_widgets_buttons ele_sel_opt =
      let pred cl =
	match ele_sel_opt with
	  None -> entity.en_ele = None
	| Some e -> data#can_append_ele e cl
      in
      List.iter
	(fun (cl, wb) ->
	  let b = pred cl in
	  if b then wb#misc#show () else wb#misc#hide ()
	)
	widgets_buttons

    method selection_changed ele =
      match selected_node with
	None ->
	  (try
	    let b = List.assq ele prop_boxes in
	    b#box#misc#hide ()
	  with Not_found -> 
	    print_DEBUG "not found dans selection changed" ;
	    ()
	  );
	  self#update_widgets_buttons None

      |	Some (g_ele, item, label, parent_opt, wt) ->
	  (try
	    let b = List.assq g_ele prop_boxes in
	    b#box#misc#show ()
	  with
	    Not_found ->
	      let b = new Zog_gui_prop.box data handler parent_opt g_ele in
	      prop_boxes <- (g_ele, b) :: prop_boxes;
	      vbox_props#pack ~expand: true b#box#coerce
	  ) ;
	  self#update_widgets_buttons (Some g_ele)
	      
(** {2 Updating the display} *)

    method insert_gui_ele ?pos wt parent_opt g_ele =
      let item = GTree.tree_item () in
      let ali = GBin.alignment ~x:0. ~xscale:0. ~packing:item#add () in
(*      let ev = GBin.event_box ~packing:ali#add () in*)
      let hbox = GPack.hbox ~packing:ali#add () in
      (match Zog_gui_icons.icon_of_gui_element_class g_ele.classe with
	None ->
	  let label_class = GMisc.label ~packing: (hbox#pack ~expand: false) 
	      ~text: ("["^(Zog_types.string_of_gui_element_class g_ele.classe) ^"]")
	      ()
	  in
	  hbox#pack ~expand: false ~padding:2 label_class#coerce
      |	Some i ->
	  let pix = Zog_gui_icons.create_pixmap i in
	  hbox#pack ~expand: false ~padding:2 pix#coerce
      );
      let label_name = GMisc.label
	  ~packing: (hbox#pack ~expand: true ~padding: 2) 
	  ~text: g_ele.name () 
      in

      handler#add_ele parent_opt g_ele label_name ;

      let _ = 
	match pos with 
	  None -> wt#append item 
	| Some n -> wt#insert item ~pos: n
      in
      (match g_ele.children with
	[] -> ()
      | _ -> 
	  let wt_sub = GTree.tree () in
	  item#set_subtree wt_sub ;
	  item#expand () ;
	  List.iter (self#insert_gui_ele wt_sub (Some g_ele)) g_ele.children ;
          if not g_ele.expanded then item#collapse () ;
      );

      ignore (item#connect#collapse (fun () -> g_ele.expanded <- false)) ;
      ignore (item#connect#expand (fun () -> g_ele.expanded <- true)) ;
      let _ = item#connect#select
	  (fun () -> 
	    print_DEBUG ("selection "^ g_ele.name);
	    selected_node <- Some (g_ele, item, label_name, parent_opt, wt) ;
	    self#selection_changed g_ele
	  )
      in
      let _ = item#connect#deselect
	  (fun () -> 
	    print_DEBUG "deselection";
	    selected_node <- None ;
	    self#selection_changed g_ele
	  ) 
      in
      (match wt#selection with
	i :: _ -> wt#unselect_item (wt#child_position i)
      |	_ -> ());
(*
      ignore (wt#select_item (wt#child_position item))
*)

    method update =
      let _ = wtree#remove_items (wtree#children) in
      List.iter 
	(fun (_, b) -> b#box#destroy ())
	prop_boxes ;
      prop_boxes <- [] ;
      (match entity.en_ele with
	None -> ()
      |	Some ele -> self#insert_gui_ele wtree None ele);
      self#update_widgets_buttons None


(** {2 Editing the entity} *)

    method insert_ele (f : unit -> gui_element) () =
       let ele = f () in
       let old_buffer = Zog_buffer.paste () in
       Zog_buffer.copy ele ;
       self#paste_append () ;
       (match old_buffer with None -> () | Some o -> Zog_buffer.copy o)

    method up () =
      match selected_node with
        None -> ()
      | Some (ele, item, _, parent_opt, tree) -> 
	  match parent_opt with
	    None -> ()
	  | Some p_ele ->
              let pos = tree#child_position item in
              if pos < 1 then
		()
              else
		(
		 data#up_element ele parent_opt ;
		 tree#remove_items [item] ;
		 handler#up_element p_ele ele ;
		 self#insert_gui_ele ~pos: (pos - 1) tree parent_opt ele ;
		 prop_boxes <- List.filter 
		     (fun (e, b) -> 
		       (e != ele) or 
		       (b#box#misc#hide () ; false))
		     prop_boxes
		)

    method set_shortcuts (window : GWindow.window) =
      (* shortcuts for inserting widgets *)
(*      let ev = new GObj.event_ops (GObj.as_widget box#coerce) in*)
      ignore (window#event#connect#key_press
		(fun ev -> 
		  print_DEBUG "key_press in gui_entity box";
		  if List.mem `CONTROL (GdkEvent.Key.state ev) then
		    let k = GdkEvent.Key.keyval ev in
		    try
		      let cl = List.assoc k widget_shortcuts in
		      try
			let (_,abb,_,_) = Zog_types.get_class_info cl in
			self#insert_ele (Zog_data.gui_element cl abb) ()
		      with
			Not_found ->
			  ()
		    with
		      Not_found -> 
(*		print_DEBUG ("shortcut string "^s^" not found");*)
			()
		  else
		    (
		     print_DEBUG "not control";
		     ()
		    );
		  true
		)
	     )



    initializer
      let _ = itemSave#connect#activate f_save in
      let _ = itemCopy#connect#activate self#copy in
      let _ = itemCut#connect#activate self#cut in
      let _ = itemPaste#connect#activate self#paste_append in
      let _ = itemDelete#connect#activate self#delete in
      let _ = itemUp#connect#activate self#up in
      let _ = wb_preview#connect#toggled
	  (fun () ->
	    Zog_misc.apply_opt
	      (fun w ->
		if wb_preview#active then w#show () else w#misc#hide ())
	      handler#window)
      in

      let tooltips = GData.tooltips () in

      let f acc (cl, s, abb, _) =
	let pix_def = 
	  match Zog_gui_icons.icon_of_gui_element_class cl with
	    None -> Zog_gui_icons.question_mark
	  | Some i -> i
	in
	let pix = Zog_gui_icons.create_pixmap pix_def in
	let wb = GButton.button () in
	wb#add pix#coerce ;
	tooltips#set_tip wb#coerce ~text: s ;
	let f_click () =
	  match cl with
	    Button ->
	      let l = [
		`I (Zog_messages.empty, self#insert_ele (Zog_data.gui_element cl abb)) ;
		`I (Zog_messages.with_label, 
		    (fun () ->
		      let e = Zog_data.gui_element cl abb () in
		      let lab = Zog_data.gui_element Label "_" () in
		      match GToolbox.input_string 
			  ~title: Zog_messages.with_label
			  (Zog_messages.enter_text^" : ")
		      with
			None -> ()
		      |	Some s ->
			  e.children <- [lab] ;
			  Zog_types.set_prop_value lab PText s ;
			  self#insert_ele (fun () -> e) ())
		   )
	      ] 
	      in
	      GToolbox.popup_menu ~entries: l ~button: 1 ~time: 0
	    | Pixmap ->
	      let l = [
		`I (Zog_messages.pixmap_file, self#insert_ele	(Zog_data.gui_element Pixmap_file abb)) ;
		`I (Zog_messages.pixmap_data, self#insert_ele	(Zog_data.gui_element Pixmap_data abb)) ;
		`I (Zog_messages.pixmap_code, self#insert_ele	(Zog_data.gui_element Pixmap_code abb)) ;
	      ] 
	      in
	      GToolbox.popup_menu ~entries: l ~button: 1 ~time: 0
	   | _ ->
	       self#insert_ele (Zog_data.gui_element cl abb) ()
	in
	ignore (wb#connect#clicked f_click) ;
	wtab_widgets#attach ~left: (acc / 18) ~top: (acc mod 18) wb#coerce ;
	widgets_buttons <- (cl, wb) :: widgets_buttons ;
	acc + 1
      in
      ignore (List.fold_left f 0 Zog_types.class_names_and_strings) ;

      (* the templates menu *)
      let f temp_entity =
	match temp_entity.en_ele with
	  None -> ()
	| Some ele ->
	    let item = GMenu.menu_item ~label: temp_entity.en_name ~packing: menuTemplates#add () in
	    ignore (item#connect#activate (self#insert_ele (fun () -> ele)))
      in
      List.iter f data#templates


  end


(** A class to handle the various windows of an entity. *)
class entity_windows data =
  object
    val mutable entities_and_windows = ([] : (entity * (gui_entity * GWindow.window)) list)

    method destroy_ent_windows ent =
      try 
	let (ent_gui, w) = List.assq ent entities_and_windows in
	entities_and_windows <- List.filter (fun (e,_) -> e != ent) entities_and_windows ;
	w#destroy () ;
	Zog_misc.apply_opt (fun w -> w#destroy ()) ent_gui#handler#window
      with Not_found -> ()

    method show_entity ent f_save =
      try
	let (ent_gui, w) = List.assq ent entities_and_windows in
	w#show () ;
	if ent_gui#preview then
	  Zog_misc.apply_opt (fun w -> w#show ()) ent_gui#handler#window
      with
	Not_found ->
	  let w = GWindow.window ~title: ent.en_name 
	      ~width: 800 ~height: 550
	      ~allow_shrink: true
	      ~allow_grow: true
	      ~auto_shrink: true ()
	  in
	  let ent_gui = new gui_entity data ent f_save in
	  let _ = w#add ent_gui#box#coerce in
	  let _ = w#add_accel_group ent_gui#accel_group in
	  let _ = w#event#connect#delete 
	      (fun _ -> 
		w#misc#hide () ; 
		Zog_misc.apply_opt (fun w -> w#misc#hide ()) ent_gui#handler#window ;
		true) 
	  in
	  entities_and_windows <- (ent, (ent_gui, w)) :: entities_and_windows ;
	  ent_gui#set_shortcuts w;
	  w#show () ;
	  ent_gui#update 

    method update_entity_name ent =
      try
	let (ent_gui, w) = List.assq ent entities_and_windows in
	w#set_title ent.en_name 
      with
	Not_found -> ()

    method update_windows =
      List.iter
	(fun (_,(b,_)) -> b#update)
	entities_and_windows
  end

(** A class to build the main box of the zoggy app. *)
class gui data =

  object (self)
    inherit Zog_gui_base.gui ()
    inherit entity_windows data as ent_wins

    val mutable selected_entity = (None : entity option)

    method init_window (w : GWindow.window) =
      w#add_accel_group accel_group

    method update_list =
      wlist_entities#clear () ;
      selected_entity <- None ;
      List.iter
	(fun ent -> 
	  ignore (wlist_entities#append
		    [ent.en_name ; (String.concat " " ent.en_params)])
	)
	data#entities;
      GToolbox.autosize_clist wlist_entities

    method update =
      self#update_list ;
      ent_wins#update_windows

    method add_entity () =
      match GToolbox.input_string Zog_messages.mnAdd_entity Zog_messages.name with
	None -> ()
      |	Some name ->
	  let ent = Zog_data.entity () in
	  ent.en_name <- name ;
	  data#add_entity ent ;
	  self#update_list

    method edit_selected_entity () =
      match selected_entity with
	None -> ()
      |	Some ent ->
	  let param_name = C.string
	      ~f: (fun s -> ent.en_name <- s)
              Zog_messages.name ent.en_name
	  in
	  let param_params = C.strings
	      ~f: (fun l -> ent.en_params <- l)
	      ~add: (fun () -> 
	               match GToolbox.input_string 
			   Zog_messages.add_parameter
			   Zog_messages.name
		       with
			 None -> []
		       | Some s -> [s]
		    )
	      Zog_messages.parameters ent.en_params
	  in
	  match C.simple_get Zog_messages.mnEntity_properties 
	      [param_name ; param_params] 
	  with
            C.Return_ok -> 
	      self#update_list ;
	      ent_wins#update_entity_name ent 
	  | _ -> ()

    method remove_selected_entity () =
      match selected_entity with
	None -> ()
      |	Some ent ->
	  ent_wins#destroy_ent_windows ent ;
	  data#remove_entity ent ;
	  self#update_list

    method up_selected_entity () =
      match selected_entity with
	None -> ()
      |	Some ent ->
	  data#up_entity ent ;
	  self#update_list

    method entity_selected (ent : entity) =
      selected_entity <- Some ent ;

    method open_selected_entity () =
      match selected_entity with
	None -> ()
      |	Some ent ->
	  ent_wins#show_entity ent (fun () -> self#save)

    method entity_deselected (ent : entity) =
      selected_entity <- None ;

    method destroy_windows =
      List.iter ent_wins#destroy_ent_windows data#entities

    method save =
      data#save ; 
      self#update

    initializer
      let maybe_double_click ev =
	let t = GdkEvent.get_type ev in
	match t with
	  `TWO_BUTTON_PRESS -> itemOpen_entity#activate ()
	| _ -> ()
      in
      let f_select ~row ~column ~event =
	try 
	  self#entity_selected (List.nth data#entities row) ;
	  (
	   match event with 
             None -> ()
           | Some ev -> maybe_double_click ev
	  )
	with Not_found -> ()
      in
      let f_unselect ~row ~column ~event =
	try 
	  self#entity_deselected (List.nth data#entities row) ;
	  (
	   match event with 
             None -> ()
           | Some ev -> maybe_double_click ev
	  )
	with Not_found -> ()
      in
      ignore (wlist_entities#connect#select_row f_select) ;
      ignore (wlist_entities#connect#unselect_row f_unselect) ;

      ignore (box#connect#destroy 
		(fun () ->
		  if data#changed then
		    (
		     match GToolbox.question_box Zog_messages.question
			 [Zog_messages.yes ; Zog_messages.no ]
			 Zog_messages.save_before_exit
		     with
		       1 -> data#save
		     | _ -> ()
		    );
		  self#destroy_windows
		)
	     );

      ignore (itemQuit#connect#activate box#destroy) ;
      ignore (itemSave#connect#activate (fun () -> self#save)) ;

      ignore (itemAdd_entity#connect#activate self#add_entity) ;
      ignore (itemOpen_entity#connect#activate self#open_selected_entity) ;
      ignore (itemUp_entity#connect#activate self#up_selected_entity) ;
      ignore (itemEntity_properties#connect#activate self#edit_selected_entity) ;
      ignore (itemRemove_entity#connect#activate self#remove_selected_entity) ;


      ignore (self#update)
  end
