(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

(** Communication. *)

open Chat_types
open Chat_proto

module M = Chat_messages
module G = Chat_global

let hook_file = ref (fun id (size,name,sock) -> ())

let open_server () =
  let socket = Unix.socket Unix.PF_INET Unix.SOCK_STREAM 0 in
  Unix.setsockopt socket Unix.SO_REUSEADDR true;
  Unix.set_nonblock socket ;
  Unix.bind socket (Unix.ADDR_INET (Unix.inet_addr_any, G.conf#port));
  Unix.listen socket 20;
  socket

let read_info host fd = 
  let chanin = Unix.in_channel_of_descr fd in
  let ((v,id,(_,port)),iddest,proto) = Chat_proto.read_info_channel chanin in
  let i = ((v,id,(host,port)),iddest,proto) in
  if v <> G.version then 
    (
     Chat_messages.verbose (Chat_messages.incompatible_versions v G.version);
     G.remove_connection (host, port);
     Unix.close fd;
     None;
    )
  else
    Some (i, fd)
       
let accept_connection socket_server =
  try
    let (sock, addr) = Unix.accept socket_server in
    let host =
      match addr with
	Unix.ADDR_INET (addr,port) -> Unix.string_of_inet_addr addr
      | _ -> failwith "not receiving data from a socket" 
    in
    Chat_messages.verbose ("receive from " ^ host) ;
    try read_info host sock
    with
    | e ->
	Unix.close sock;
	raise e
  with
    Unix.Unix_error (Unix.EWOULDBLOCK,_,_)
  | Unix.Unix_error (Unix.EAGAIN,_,_) ->
      None
  | Unix.Unix_error (e,s1,s2) ->
      let s = (Unix.error_message e)^" :"^s1^" "^s2 in
      raise (Failure s)
  | e ->
      let s = Printexc.to_string e in
      raise (Failure s)

let rec receive socket_server =
  try
    let conns = G.connections_list () in
(*    Chat_messages.verbose "Unix.select...";*)
    let (ready,_,_) = Unix.select (conns @ [socket_server]) [] [] 0.0 in
(*    Chat_messages.verbose "Unix.select done";*)
    match ready with
      [] -> None
    | h :: q when h = socket_server ->
	accept_connection socket_server	
    | h :: q ->
	let l = G.socket_info_list () in
	let (id,host,port) = List.assoc h l in
	Chat_messages.verbose 
	  ("message from a connected : "^host);
	try
	  read_info host h
	with
	  e ->
	    G.remove_connection (host,port);
	    Unix.close h;
	    prerr_endline (Printexc.to_string e);
	    None
  with
    Unix.Unix_error (Unix.EWOULDBLOCK,_,_) -> None
  | Unix.Unix_error (Unix.EAGAIN,_,_)
  | Unix.Unix_error (Unix.EINTR,_,_) -> 
(*      prerr_endline "interrupted";*)
      receive socket_server
  | Unix.Unix_error (_,_,_) ->
      G.check_connections ();
      None

(** Return [true] if we got connected. *)
let do_connect sock sockaddr =
  Unix.set_nonblock sock ;
  try
    Unix.connect sock sockaddr; 
    Unix.clear_nonblock sock ;
    true
  with
    Unix.Unix_error (e, _,_ ) ->
      match e with 
	Unix.EWOULDBLOCK | Unix.EINPROGRESS ->
	  let res = 
	    match Unix.select [] [sock] [] G.conf#second_send_timeout with
	    | (_, [s], _) when s = sock-> 
		(
		 try ignore(Unix.getpeername s) ; true
		 with Unix.Unix_error (_, _,_ ) -> false
		)
	    | _ -> false
	  in
	  Unix.clear_nonblock sock ;
	  res
      | _ ->
	  false

let create_connection adr =
  let (host, port) = adr in
  let sock = Unix.socket Unix.PF_INET Unix.SOCK_STREAM 0 in
  prerr_endline ("create_connection: host="^host);
  prerr_endline ("create_connection: ip_of_host host="^(G.ip_of_host host));
  let ip = Unix.inet_addr_of_string (G.ip_of_host host) in
  let sockaddr = Unix.ADDR_INET (ip, port) in
  try
    if do_connect sock sockaddr then
      sock
    else
       raise (Failure (M.could_not_connect host port))
  with
  | Unix.Unix_error (e,s1,s2) ->
      let s = s1^" "^s2^" : "^(Unix.error_message e) in
      raise (Failure s)

let get_connection ?(temp=false) iddest adr =
  if not temp then
    try
      fst (G.find_connection adr)
    with
      Not_found ->
	let sock = create_connection adr in
	G.add_connection adr sock iddest;
	sock
  else
    create_connection adr

let send ?sock iddest (host,port) proto =
  try
    let sock = 
      match sock with
	None -> get_connection iddest (host,port) 
      |	Some s -> s
    in
    let oc = Unix.out_channel_of_descr sock in
    let i = (G.source (), iddest, proto) in
    try
      Chat_proto.write_info_channel oc i
    with
    | Sys_error s ->
	G.remove_connection (host,port);
	G.safe_close sock
  with
    Failure s ->
      prerr_endline s
  | e ->
      prerr_endline (Printexc.to_string e)

let handle_info id host port proto sock =
  match proto with
  | Hello -> G.add_connection (host, port) sock id
  | HelloOk -> ()
  | Byebye -> 
      Chat_messages.verbose ("Byebye from "^host^". Closing connection.");
      G.remove_connection (host, port) ;
      Unix.close sock
  | Message _ ->
      G.add_connection (host, port) sock id
  | RoomMessage (_,l,_) ->
      G.add_connection (host, port) sock id
  | AddOpen _ -> 
      () (* A VOIR *)
  | File (size,name) ->
      ignore (Thread.create (!hook_file id) (size,name,sock))
