(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

(** OCamldoc latex generator handling command custom tags *)

open Odoc_info

open Odoc_info.Value
open Odoc_info.Module

let opt = Odoc_info.apply_opt

let tables_only = ref false

class gen () =
  object (self)
    inherit Odoc_latex.latex as latex

    method generate_for_module chanout m =
      let depth = Name.depth m.m_name in
      try
	Odoc_info.verbose ("Generate for module "^m.m_name);
	if not !tables_only then
	  (
	   let (first_t, rest_t) = self#first_and_rest_of_info m.m_info in
	   let text = [ Title (depth, None,
			       [ Raw (Odoc_messages.modul^" ") ; Code m.m_name ] @
			       (match first_t with 
				[] -> []
			       | t -> (Raw " : ") :: t)) ;
			Latex (self#make_label m.m_name) ;
		      ] 
	   in
	   output_string chanout (self#latex_of_text text);
	   output_string chanout ("\n\n"^(self#latex_of_text rest_t))
	  )
	else
	  ();

	let vals = Odoc_info.Search.values [m] in
	let mets = Odoc_info.Search.methods [m] in
	let rec iter tag f_info = function
	    [] -> []
	  | v :: q ->
	      match f_info v with
		None -> iter tag f_info q
	      |	Some i ->
		  try
		    let t = List.assoc tag i.i_custom in
		    (t, i.i_desc) :: (iter tag f_info q)
		  with
		    Not_found ->
		      iter tag f_info q
	in
	let commands_pre =
	  (iter "command" (fun v -> v.val_info) vals) @
	  (iter "command" (fun m -> m.met_value.val_info) mets)
	in
	let commands = 
	  List.sort 
	    (fun (t1,_) (t2,_) -> 
	      compare (Odoc_info.string_of_text t1) (Odoc_info.string_of_text t2))
	    commands_pre
	in	    
	(
	 match commands with
	   [] ->
	     ()
	 | l ->
	     output_string chanout (self#latex_of_Title (depth+1) None [Raw "Commands"]);
	     output_string chanout "\n\\begin{tabular}{|p{3.5cm}|p{13cm}|}\n \\hline\n";
	     output_string chanout "{\\bf Command} & {\\bf Description} \\\\ \\hline\n";
	     List.iter
	       (fun (t, d) ->
		 output_string chanout (self#latex_of_text t);
		 output_string chanout "&";
		 (match d with
		   None -> ()
		 | Some desc -> output_string chanout (self#latex_of_text desc));
		 output_string chanout "\\\\ \\hline\n"
	       )
	       commands;
	     output_string chanout "\\end{tabular}\n";
	);

	let editors_pre =
	  (iter "editor" (fun v -> v.val_info) vals) @
	  (iter "editor" (fun m -> m.met_value.val_info) mets)
	in
	let editors = 
	  List.sort 
	    (fun (t1,_) (t2,_) -> 
	      compare (Odoc_info.string_of_text t1) (Odoc_info.string_of_text t2))
	    editors_pre
	in	    
	(
	 match editors with
	   [] ->
	     ()
	 | l ->
	     output_string chanout (self#latex_of_Title (depth + 1) None [Raw "Editors"]);
	     output_string chanout "\n\\begin{tabular}{|l|l|}\n \\hline\n";
	     output_string chanout "{\\bf Editor} & {\\bf Description} \\\\ \\hline\n";
	     List.iter
	       (fun (t, d) ->
		 output_string chanout (self#latex_of_text t);
		 output_string chanout "&";
		 (match d with
		   None -> ()
		 | Some desc -> output_string chanout (self#latex_of_text desc));
		 output_string chanout "\\\\ \\hline\n"
	       )
	       editors;
	     output_string chanout "\\end{tabular}\n";

	);
	
      with
	Sys_error s ->
	  prerr_endline s

  end

let generator = ((new gen ()) :> Odoc_args.doc_generator)

let op_tables_only = ("-tables-only", Arg.Set tables_only, "Generate code for tables only, no index, no module description")

let _ = Odoc_args.add_option op_tables_only

let _ = Odoc_args.set_doc_generator (Some generator)
