/*
 * wrapper.cxx
 *
 * OpenH323 Wrapper Library
 *
 * Copyright (c) 2002-2005 InAccess Networks
 * Michalis Manousos <manousos@inaccessnetworks.com>
 * Dimitris Economou <decon@inaccessnetworks.com>
 *
 * This file is part of "H.323 support for ASTERISK"
 *
 * "H.323 support for ASTERISK" is free software;
 * you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation;
 * either version 2 of the License, or (at your option) any later version. 
 *
 * "H.323 support for ASTERISK" is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details. 
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 * $Id: wrapper.cxx,v 1.64.2.3 2005/01/17 18:11:50 manousos Exp $
 *
 */

/************************************************************************/
/* INCLUDE FILES ********************************************************/

#include <ptlib.h>
#include <h323.h>
#include <h323pdu.h>
#include <mediafmt.h>
#include <lid.h>
#include <gkserver.h>

#include <list>
#include <string>
#include <algorithm>

#define cplusplus
#include "version.h"
#include "wrapcaps.hxx"
#include "wrapper_misc.hxx"
#include "wrapper.hxx"
#include "wrapendpoint.hxx"
#include "wrapconnection.hxx"
#include "wrapgkserver.hxx"
#include "asteriskaudio.hxx"

using namespace std;

/************************************************************************/
/* Wrapper's internal variables and classes *****************************/

#ifdef	WRAPTRACING
int wrapTraceLevel;
#endif

/* These are set by the application */
static int version_maj = 0;
static int version_min = 0;
static int version_bld = 0;
static char app_name[128] = { 0 };

/* Options for connections creation */
BOOL	Con_noFastStart;
BOOL	Con_noH245Tunnelling;
BOOL	Con_noH245inSetup;
BOOL	Con_noInBandDTMF;
WORD	Con_jitterMin;
WORD	Con_jitterMax;
BOOL	Con_noSilenceSuppression;

/* Callback functions */
start_logchan_cb	on_start_logical_channel; 
clear_con_cb		on_connection_cleared;
alert_con_cb		on_connection_alert;
h323_exception_cb	on_h323_exception;
init_con_cb			on_connection_init;
stats_con_cb		on_connection_stats;

/**
 * The wrapper library assumes that only one endpoint should exist.
 * The application cannot run twice the h323_end_point_create(). 
 */
WrapH323EndPoint *endPoint = NULL;

/**
 * The gatekeeper server object.
 */
WrapGatekeeperServer *gkServer = NULL;


/**************************************************************************
 * The internal class for holding the listener and its
 * description 
 */
class WrapListener {

	public:
	H323Listener *listener;
	lis_type_t listenerType;
    
	/* Constructor */
	WrapListener(H323Listener *l, lis_type_t t)
	{
		listener = l;
		listenerType = t;
	};
};

/**
 * A list with the H323 listeners of the endpoint 
 */
list<WrapListener*> listenerList; 


/**************************************************************************
 * The WrapThread is a necessary descendant of PProcess class so that the H323EndPoint 
 * object to be created from within that class. The reason is that the constructor
 * of the H323EndPoint class needs to find some information related to a hypothetically
 * PProcess class which is supposed to instantiates it. 
 */
class WrapProcess : public PProcess {

	PCLASSINFO(WrapProcess, PProcess);

	public:
	WrapProcess(char **gwprefix_tab, int gwprefix_num, int logLev, char *logFile)
	: PProcess("inAccess Networks (www.inaccessnetworks.com)", app_name, 
		version_maj, version_min, ReleaseCode, version_bld)
	{ 
		endPoint = NULL;
		prefixes = gwprefix_tab;
		prefix_num = gwprefix_num;
		log_lev = logLev;
		if ((logFile == NULL) || (logFile[0] == '\0'))
			log_file = NULL;
		else
			log_file = logFile;
		Resume(); 
	};

	~WrapProcess()
	{
		WRAPTRACE(4, "Going down.");
		if (endPoint != NULL) {
			delete endPoint;
			endPoint = NULL;
		}
		if (gkServer != NULL) {
			delete gkServer;
			gkServer = NULL;
		}
		PTrace::SetLevel(0);
	}
    
	void 
	Main()
	{
		PTrace::Initialise(log_lev, log_file);
		endPoint = new WrapH323EndPoint(prefixes, prefix_num);
		gkServer = new WrapGatekeeperServer((H323EndPoint &)*endPoint);
	};

	protected:
	char **prefixes;
	int prefix_num;
	int log_lev;
	char *log_file;
};
WrapProcess *localProcess = NULL;


/**************************************************************************
 * This class handles the termination of a call.
 * Note that OpenH323 Library requires that the termination
 * of a call should be done inside a separate thread of execution.
 */
class ClearCallThread : public PThread {

	PCLASSINFO(ClearCallThread, PThread);

	public:
	ClearCallThread(const char *tc)
		: PThread(10000, PThread::AutoDeleteThread)
	{ 
		WRAPTRACE(4, "Object initialized.");
		WRAPTRACE(4, "Unblock pipe - " << unblockPipe[0] << ", " << unblockPipe[1]);
		token = tc;
		Resume(); 
	};

	~ClearCallThread()
	{
		WRAPTRACE(4, "Object deleted.");
		return;
	};
    
	void 
	Main()
	{
		//endPoint->ClearCall(token);
		//if (endPoint->ClearCallSynchronous(token) == FALSE) {
		if (endPoint->ClearCall(token) == FALSE) {
			WRAPTRACE(2, "Failed to clear call with token " << token);
		} else {
			WRAPTRACE(2, "Call with token " << token << " cleared.");
		}
		return;
	};

	protected:
	PString	token;
};

/**************************************************************************
 * This class handles the registration with a gatekeeper.
 */
class GKRegThread : public PThread {

	PCLASSINFO(GKRegThread, PThread);

	public:
	GKRegThread(int mode, const char *gkname)
		: PThread(10000)
	{ 
		WRAPTRACE(4, "Object initialized.");
		WRAPTRACE(4, "Unblock pipe - " << unblockPipe[0] << ", " << unblockPipe[1]);
		gkName = gkname;
		gkMode = mode;
	};

	~GKRegThread()
	{
		WRAPTRACE(4, "Object deleted.");
		return;
	};
    
	void 
	Main()
	{
		switch (gkMode) {
			case GKMODE_DISABLE:
				break;

			case GKMODE_DISCOVER:
				WRAPTRACE(2, "Searching for gatekeeper...");
				if (endPoint->DiscoverGatekeeper())
					WRAPTRACE(2, "Gatekeeper found.");
				else
					WRAPTRACE(2, "Could not find a gatekeeper.");
				break;

			case GKMODE_NAME:
				WRAPTRACE(2, "Setting gatekeeper...");
				if (endPoint->SetGatekeeper(gkName))
					WRAPTRACE(2, "Gatekeeper found.");
				else
					WRAPTRACE(2, "Error registering with gatekeeper '" 
							<< gkName << "'.");
				break;

			case GKMODE_ID:
				WRAPTRACE(2, "Locating gatekeeper...");
				if (endPoint->LocateGatekeeper(gkName))
					WRAPTRACE(2, "Gatekeeper found.");
				else
					WRAPTRACE(2, "Error registering with gatekeeper '"
							<< gkName << "'.");
				break;

			default:
				WRAPTRACE(2, "Unknown gatekeeper mode (" << gkMode << ").");
				break;
		}
		return;
	};

	protected:
	PString	gkName;
	int gkMode;
};
GKRegThread *gkRegThread = NULL;


/************************************************************************/
/* INTERNAL C-FUNCTIONS OF WRAPPER **************************************/

/**************************************************************************
 * Creates a H323Capability based on the given capability/codec type.
 * Returns a pointer to the capability on success, NULL on failure
 * or unsupported capability.
 */
H323Capability *
h323_capability_create(WrapH323EndPoint *ep, cap_type_t capability, int frame_num)
{
	H323Capability	*newCap;
	int frames;

	newCap = NULL;
	switch (capability) {
		case G711U:
		{
			// Use the built-in codec for G.711 u-law
			//H323_G711Capability *g711muCap = new H323_G711Capability(H323_G711Capability::muLaw);
			H323_LIDCapability *g711muCap = new H323_LIDCapability(OPAL_G711_ULAW_64K);
			if (frame_num > 0)
				ep->SetFrames(OpalG711uLaw, frame_num);
			frames = ep->GetFrames(OpalG711uLaw);
			if (frames <= 0) {
				delete g711muCap;
				break;
			}
			g711muCap->SetTxFramesInPacket(frames);
			newCap = g711muCap;
			break;
		}
		case G711A:
		{
			// Use the built-in codec for G.711 A-law
			//H323_G711Capability *g711aCap = new H323_G711Capability(H323_G711Capability::ALaw);
			H323_LIDCapability *g711aCap = new H323_LIDCapability(OPAL_G711_ALAW_64K);
			if (frame_num > 0)
				ep->SetFrames(OpalG711ALaw, frame_num);
			frames = ep->GetFrames(OpalG711ALaw);
			if (frames <= 0) {
				delete g711aCap;
				break;
			}
			g711aCap->SetTxFramesInPacket(frames);
			newCap = g711aCap;
			break;
		}
		case G7231:
		case G72316K3:
		{
			// G.723.1(6.3k) pass-through codec
			H323_LIDCapability *g72316k3Cap = new H323_LIDCapability(OPAL_G7231_6k3);
			if (frame_num > 0)
				ep->SetFrames(OpalG7231_6k3, frame_num);
			frames = ep->GetFrames(OpalG7231_6k3);
			if (frames <= 0) {
				delete g72316k3Cap;
				break;
			}
			g72316k3Cap->SetTxFramesInPacket(frames);
			newCap = g72316k3Cap;
			break;
		}
		case G72315K3:
		{
			// G.723.1(5.3k) pass-through codec
			H323_LIDCapability *g72315k3Cap = new H323_LIDCapability(OPAL_G7231_5k3);
			if (frame_num > 0)
				ep->SetFrames(OpalG7231_5k3, frame_num);
			frames = ep->GetFrames(OpalG7231_5k3);
			if (frames <= 0) {
				delete g72315k3Cap;
				break;
			}
			g72315k3Cap->SetTxFramesInPacket(frames);
			newCap = g72315k3Cap;
			break;
		}
		case G7231A6K3:
		{
			// G.723.1A(6.3k) pass-through codec
			H323_LIDCapability *g7231A6k3Cap = new H323_LIDCapability(OPAL_G7231A_6k3);
			if (frame_num > 0)
				ep->SetFrames(OpalG7231A_6k3, frame_num);
			frames = ep->GetFrames(OpalG7231A_6k3);
			if (frames <= 0) {
				delete g7231A6k3Cap;
				break;
			}
			g7231A6k3Cap->SetTxFramesInPacket(frames);
			newCap = g7231A6k3Cap;
			break;
		}
		case G7231A5K3:
		{
			// G.723.1A(5.3k) pass-through codec
			H323_LIDCapability *g7231A5k3Cap = new H323_LIDCapability(OPAL_G7231A_5k3);
			if (frame_num > 0)
				ep->SetFrames(OpalG7231A_5k3, frame_num);
			frames = ep->GetFrames(OpalG7231A_5k3);
			if (frames <= 0) {
				delete g7231A5k3Cap;
				break;
			}
			g7231A5k3Cap->SetTxFramesInPacket(frames);
			newCap = g7231A5k3Cap;
			break;
		}
		case G72616K:
		{
			// G.726(16k) pass-through codec
			Wrap_G726_Capability *g72616kCap = 
							new Wrap_G726_Capability(WRAP_G726_SPEED16);
			if (frame_num > 0)
				ep->SetFrames(WrapG726_16, frame_num);
			frames = ep->GetFrames(WrapG726_16);
			if (frames <= 0) {
				delete g72616kCap;
				break;
			}
			g72616kCap->SetTxFramesInPacket(frames);
			newCap = g72616kCap;
			break;
		}
		case G72624K:
		{
			// G.726(24k) pass-through codec
			Wrap_G726_Capability *g72624kCap = 
							new Wrap_G726_Capability(WRAP_G726_SPEED24);
			if (frame_num > 0)
				ep->SetFrames(WrapG726_24, frame_num);
			frames = ep->GetFrames(WrapG726_24);
			if (frames <= 0) {
				delete g72624kCap;
				break;
			}
			g72624kCap->SetTxFramesInPacket(frames);
			newCap = g72624kCap;
			break;
		}
		case G726:
		case G72632K:
		{
			// G.726(32k) pass-through codec
			Wrap_G726_Capability *g72632kCap = 
							new Wrap_G726_Capability(WRAP_G726_SPEED32);
			if (frame_num > 0)
				ep->SetFrames(WrapG726_32, frame_num);
			frames = ep->GetFrames(WrapG726_32);
			if (frames <= 0) {
				delete g72632kCap;
				break;
			}
			g72632kCap->SetTxFramesInPacket(frames);
			newCap = g72632kCap;
			break;
		}
		case G72640K:
		{
			// G.726(40k) pass-through codec
			Wrap_G726_Capability *g72640kCap = 
							new Wrap_G726_Capability(WRAP_G726_SPEED40);
			if (frame_num > 0)
				ep->SetFrames(WrapG726_40, frame_num);
			frames = ep->GetFrames(WrapG726_40);
			if (frames <= 0) {
				delete g72640kCap;
				break;
			}
			g72640kCap->SetTxFramesInPacket(frames);
			newCap = g72640kCap;
			break;
		}
		case G728:
		{
			// G.728 pass-through codec
			H323_LIDCapability *g728Cap = new H323_LIDCapability(OPAL_G728);
			if (frame_num > 0)
				ep->SetFrames(OpalG728, frame_num);
			frames = ep->GetFrames(OpalG728);
			if (frames <= 0) {
				delete g728Cap;
				break;
			}
			g728Cap->SetTxFramesInPacket(frames);
			newCap = g728Cap;
			break;
		}
		case G729:
		{
			// G.729 pass-through codec
			H323_LIDCapability *g729Cap = new H323_LIDCapability(OPAL_G729);
			if (frame_num > 0)
				ep->SetFrames(OpalG729, frame_num);
			frames = ep->GetFrames(OpalG729);
			if (frames <= 0) {
				delete g729Cap;
				break;
			}
			g729Cap->SetTxFramesInPacket(frames);
			newCap = g729Cap;
			break;
		}
		case G729A:
		{
			// G.729A pass-through codec
			H323_LIDCapability *g729ACap = new H323_LIDCapability(OPAL_G729A);
			if (frame_num > 0)
				ep->SetFrames(OpalG729A, frame_num);
			frames = ep->GetFrames(OpalG729A);
			if (frames <= 0) {
				delete g729ACap;
				break;
			}
			g729ACap->SetTxFramesInPacket(frames);
			newCap = g729ACap;
			break;
		}
		case G729B:
		{
			// G.729B pass-through codec
			H323_LIDCapability *g729BCap = new H323_LIDCapability(OPAL_G729B);
			if (frame_num > 0)
				ep->SetFrames(OpalG729B, frame_num);
			frames = ep->GetFrames(OpalG729B);
			if (frames <= 0) {
				delete g729BCap;
				break;
			}
			g729BCap->SetTxFramesInPacket(frames);
			newCap = g729BCap;
			break;
		}
		case G729AB:
		{
			// G.729AB pass-through codec
			H323_LIDCapability *g729ABCap = new H323_LIDCapability(OPAL_G729AB);
			if (frame_num > 0)
				ep->SetFrames(OpalG729AB, frame_num);
			frames = ep->GetFrames(OpalG729AB);
			if (frames <= 0) {
				delete g729ABCap;
				break;
			}
			g729ABCap->SetTxFramesInPacket(frames);
			newCap = g729ABCap;
			break;
		}
		case GSM0610:
		{
			// GSM 06.10 pass-through codec
			H323_LIDCapability *gsmCap = new H323_LIDCapability(OPAL_GSM0610);
			if (frame_num > 0)
				ep->SetFrames(OpalGSM0610, frame_num);
			frames = ep->GetFrames(OpalGSM0610);
			if (frames <= 0) {
				delete gsmCap;
				break;
			}
			gsmCap->SetTxFramesInPacket(frames);
			newCap = gsmCap;
			break;
		}
		case MSGSM:
		{
#if 0
			// Use the built-in codec for Microsoft GSM
			MicrosoftGSMAudioCapability * msGsmCap = new MicrosoftGSMAudioCapability;
			msGsmCap->SetTxFramesInPacket(4);
			newCap = msGsmCap;
#endif
			break;
		}
		case LPC10:
		{
#if 0
			// Use the built-in codec for LPC10
			if (ep == NULL)
				break;
			H323_LPC10Capability * lpcCap = new H323_LPC10Capability(*ep);
			newCap = lpcCap;
#endif
			break;
		}
		default:
			// Unknown codec!
			break;
	}
	return(newCap);
}

/************************************************************************/
/* IMPLEMENTATION OF WRAPPER API ****************************************/

/**
 * The extern "C" directive takes care for 
 * the ANSI-C representation of linkable symbols 
 */
extern "C" {

	static struct {
		int h323_reason;
		int wrap_reason;
		char *desc;
	} call_end_reason[] = {
		{ H323Connection::EndedByLocalUser, OH323END_LOCAL_USER, "Cleared by local user" },
		{ H323Connection::EndedByNoAccept, OH323END_NO_ACCEPT, "Incoming call was not accepted" },
		{ H323Connection::EndedByAnswerDenied, OH323END_ANSWER_DENIED, "Incoming call denied" },
		{ H323Connection::EndedByRemoteUser, OH323END_REMOTE_USER, "Cleared by remote user" },
		{ H323Connection::EndedByRefusal, OH323END_REFUSAL, "Remote user refused call" },
		{ H323Connection::EndedByNoAnswer, OH323END_NO_ANSWER, "Remote user did not answer call" },
		{ H323Connection::EndedByCallerAbort, OH323END_CALLER_ABORT, "Remote user stopped calling" },
		{ H323Connection::EndedByTransportFail, OH323END_TRANSPORT_FAIL, "Transport failure" },
		{ H323Connection::EndedByConnectFail, OH323END_CONNECT_FAIL, "Connection failure" },
		{ H323Connection::EndedByGatekeeper, OH323END_GATEKEEPER, "Gatekeeper cleared call" },
		{ H323Connection::EndedByNoUser, OH323END_NO_USER, "Gatekeeper could not find user" },
		{ H323Connection::EndedByNoBandwidth, OH323END_NO_BANDWIDTH, "Not enough bandwidth" },
		{ H323Connection::EndedByCapabilityExchange, OH323END_CAPABILITY, "No common codecs" },
		{ H323Connection::EndedByCallForwarded, OH323END_CALLFWD, "Call was forwarded" },
		{ H323Connection::EndedBySecurityDenial, OH323END_SECURITY, "Call ended due to security checks" },
		{ H323Connection::EndedByLocalBusy, OH323END_LOCAL_BUSY, "Local endpoint is busy" },
		{ H323Connection::EndedByLocalCongestion, OH323END_LOCAL_CONGESTION, "Local endpoint is congested" },
		{ H323Connection::EndedByRemoteBusy, OH323END_REMOTE_BUSY, "Remote endpoint is busy" },
		{ H323Connection::EndedByRemoteCongestion, OH323END_REMOTE_CONGESTION, "Remote endpoint is congested" },
		{ H323Connection::EndedByUnreachable, OH323END_UNREACHABLE, "Remote endpoint is unreachable" },
		{ H323Connection::EndedByNoEndPoint, OH323END_NO_ENDPOINT, "No endpoint" },
		{ H323Connection::EndedByHostOffline, OH323END_HOST_OFFLINE, "Remote endpoint is offline" },
		{ H323Connection::EndedByTemporaryFailure, OH323END_TEMP_FAILURE, "Temporary failure" },
		{ H323Connection::EndedByQ931Cause, OH323END_Q931CAUSE, "Call ended with Q.931 cause" },
		{ H323Connection::EndedByDurationLimit, OH323END_DURATION_LIMIT, "Call ended due to enforced duration limit" },
		{ -1, -1, "Unknown reason" },
	};

	/**************************************************************************
	 * Check whether the H.323 endpoint has already been created
	 */
	boolean_t 
	end_point_exist(void)
	{
		if (endPoint == NULL) {
			return NO;
		}
		return YES;
	}
 
	/**************************************************************************
     *
     */
	void
	h323_appinfo_set(char *name, int major, int minor, int build)
	{
		memset(app_name, 0, sizeof(app_name));
		strncpy(app_name, name, sizeof(app_name) - 1);
		version_maj = major;
		version_min = minor;
		version_bld = build;
	}

	/**************************************************************************
	 * Create the H.323 endpoint
	 */
	void 
	h323_end_point_create(char **gwprefix_tab, int gwprefix_num, 
			int wrap_log_lev, int log_lev, char *log_file)
	{
		if (end_point_exist() == YES) {
			WRAPTRACEAPI(1, "Endpoint exists! Destroy it first.");
			return;
		}

		on_start_logical_channel = NULL;
		on_connection_cleared = NULL;
		on_connection_alert = NULL;
		on_h323_exception = NULL;
		on_connection_init = NULL;
		on_connection_stats = NULL;
		channelsOpen = 0;
#ifdef	WRAPTRACING
		wrapTraceLevel = wrap_log_lev;
#endif
		localProcess = new WrapProcess(gwprefix_tab, gwprefix_num, log_lev, log_file);
		localProcess->Main();
		WRAPTRACEAPI(2, "Endpoint created.");
	}

	/**************************************************************************
	 * Destroy the H.323 endpoint
	 */
	void 
	h323_end_point_destroy(void)
	{
		WRAPTRACEAPI(2, "Destroying endpoint.");
		if (end_point_exist() == NO) {
			return;
		}

		/* Cleanup stuff */
		//endPoint->ClearAllCalls();

		if (gkRegThread != NULL) {
			gkRegThread->WaitForTermination();
			delete gkRegThread;
			gkRegThread = NULL;
		}

		if (localProcess != NULL) {
			delete localProcess;
			localProcess = NULL;
		}
	}

	/**************************************************************************
	 * Install the callback functions
	 */
	int 
	h323_callback_register(start_logchan_cb sfunc, clear_con_cb cfunc,
						alert_con_cb alertfunc, h323_exception_cb exfunc,
						init_con_cb initfunc, stats_con_cb statsfunc)
	{
		on_start_logical_channel = sfunc;
		on_connection_cleared = cfunc;
		on_connection_alert = alertfunc;
		on_h323_exception = exfunc;
		on_connection_init = initfunc;
		on_connection_stats = statsfunc;
		WRAPTRACEAPI(3, "Callback functions installed.");
		return(0);
	}

	/**************************************************************************
	 * Add the specified capability to the capabilities table of the H.323 endpoint 
	 * Note: AddCapability puts the codec into the list of codecs we can send
	 */
	cap_ret_val_t
	h323_add_capability(cap_type_t cap, int frames)
	{
		H323Capability					*h323Cap;
		cap_ret_val_t					ret_val;

		if (end_point_exist() == NO) {
			return CAP_EP_ER;
		}

		ret_val = CAP_NSUP_ER;

		/* Create and add the capability */
		h323Cap = h323_capability_create(endPoint, cap, frames);
		if (h323Cap != NULL) {
			endPoint->AddCapability(h323Cap);
			WRAPTRACEAPI(2, "Added capability " << h323Cap->GetFormatName());
			ret_val = CAP_INSERT_OK;
		} else {
			WRAPTRACEAPI(2, "Failed to add capability type " << cap);
			ret_val = CAP_NSUP_ER;
		}
		return ret_val;
	}

	/**************************************************************************
	 * Set the specified capability as the first capability of the H.323 endpoint
	 * Note: SetCapability puts the codec into the list of codecs we can send and receive
	 */
	cap_ret_val_t 
	h323_set_capability(cap_type_t cap, int frames)
	{
		H323Capability					*h323Cap;
		cap_ret_val_t					ret_val;

		if (end_point_exist() == NO) {
			return CAP_EP_ER;
		}

		ret_val = CAP_NSUP_ER;

		/* Create and add the capability */
		h323Cap = h323_capability_create(endPoint, cap, frames);
		if (h323Cap != NULL) {
			endPoint->SetCapability(0, 0, h323Cap);
			//endPoint->SetCapability(P_MAX_INDEX, P_MAX_INDEX, h323Cap);
			WRAPTRACEAPI(2, "Inserted capability " << h323Cap->GetFormatName());
			ret_val = CAP_INSERT_OK;
		} else {
			WRAPTRACEAPI(2, "Failed to insert capability type " << cap);
			ret_val = CAP_NSUP_ER;
		}
		return ret_val;
    }

	/**************************************************************************
	 * Set the mode for sending/receiving user-input (DTMF).
	 */
	cap_ret_val_t
	h323_set_senduimode(uimode_t cap)
	{
		if (end_point_exist() == NO) {
			return CAP_EP_ER;
		}

		/* Set the specified send user-input mode */
		switch (cap) {
			case UIMODE_Q931:
				endPoint->SetSendUserInputMode(H323Connection::SendUserInputAsQ931);
				break;
			case UIMODE_STRING:
				endPoint->SetSendUserInputMode(H323Connection::SendUserInputAsString);
				break;
			case UIMODE_TONE:
				endPoint->SetSendUserInputMode(H323Connection::SendUserInputAsTone);
				break;
			case UIMODE_RFC2833:
				endPoint->SetSendUserInputMode(H323Connection::SendUserInputAsInlineRFC2833);
				break;
			default:
				return CAP_NSUP_ER;
		}

		/* Add all user input capabilities (receiving user-input) */
		endPoint->AddAllUserInputCapabilities(0, P_MAX_INDEX);

		WRAPTRACEAPI(3, "User-input mode set.");
		return CAP_INSERT_OK;
	}

	/**************************************************************************
	 * The check_type function is a logical predicate function
	 * for the find_if algorithm which is used in the h323_remove_listener
	 * as well as in the h323_start_listener
	 */
	static lis_type chkLis;

	static bool
	check_lis_type(WrapListener *wl)
	{
		if (wl->listenerType == chkLis) {
			return true;
		} else {
			return false;
		}
	}

	/**************************************************************************
	 * Start the H.323 listener
	 */
	lis_ret_val_t
	h323_start_listener(lis_type lis, char *listenAddress, int listenPort)
	{
		H323Listener					*newListener;
		lis_type_t						listenerType;
		list<WrapListener*>::iterator	cit;
		lis_ret_val_t					ret_val;

		if (end_point_exist() == NO) {
			return LIS_EP_ER;
		}

		ret_val = LIS_NSUP_ER;

		/* Check to see if a same type of listener exists in the listeners list */
		chkLis = lis;
		cit = find_if(listenerList.begin(), listenerList.end(), check_lis_type);

		if ((cit != listenerList.end())) {
			ret_val = LIS_EXIST;
			goto exit;
		}

		newListener = NULL;
		switch (lis) {
			case TCP:
			{
				PIPSocket::Address interfaceAddress(listenAddress);
				H323ListenerTCP *tcpListener = new H323ListenerTCP(*endPoint, interfaceAddress, (WORD)listenPort);
				newListener = tcpListener;
				listenerType = TCP;
				break;
			}
			case UDP:
				listenerType = UDP;
				break;
			default:
				listenerType = LIS_UNDEFINED;
				break;
		}

		/* if a new listener is created then setup stuff */
		if (newListener != NULL) {	
			if (!endPoint->StartListener(newListener)) {
				WRAPTRACEAPI(2, "Could not open H.323 listener on "
				     << ((H323ListenerTCP *) newListener));
				ret_val = LIS_FAILOP_ER;
				goto exit;
			}
			WRAPTRACEAPI(2, "Started listener " << *((H323ListenerTCP *) newListener));

			WrapListener *wl = new WrapListener(newListener, listenerType);

			/* Insert the listener to the listeners list */
			listenerList.insert(listenerList.end(), wl);
		} else {
			WRAPTRACEAPI(2, "Unsupported listener type (%d).");
			goto exit;
		}

		ret_val = LIS_START_OK;
	exit:
		return ret_val;
	}

	/**************************************************************************
	 * Set the port ranges for the H.323 endpoint
	 */
	int
	h323_set_ports(unsigned tcpBase, unsigned tcpMax, unsigned udpBase, unsigned udpMax,
				unsigned rtpBase, unsigned rtpMax)
	{
		WRAPTRACEAPI(3, "Setting endpoint port ranges.");

		if (end_point_exist() == NO)
			return -1;

		endPoint->SetTCPPorts(tcpBase, tcpMax);
		endPoint->SetUDPPorts(udpBase, udpMax);
		endPoint->SetRtpIpPorts(rtpBase, rtpMax);
		return 0;
	}

	/**************************************************************************
	 * Configure the H.323 endpoint
	 */
	int
	h323_set_options(int noFastStart, int noH245Tunnelling, int noH245inSetup,
				int noSilenceSuppression, int bwLimit, int jitterMin, int jitterMax,
#ifdef HAS_OH323MODS
				int jitterDec, int jitterInc, 
#endif
				int ipTos)
	{
		WRAPTRACEAPI(3, "Setting endpoint options.");

		if (end_point_exist() == NO)
			return -1;

		WRAPTRACEAPI(3, "  FastStart option.");
		if (noFastStart)
			endPoint->DisableFastStart(TRUE);
		else
			endPoint->DisableFastStart(FALSE);
		WRAPTRACEAPI(3, "  H245Tunnelling option.");
		if (noH245Tunnelling)
			endPoint->DisableH245Tunneling(TRUE);
		else
			endPoint->DisableH245Tunneling(FALSE);
		WRAPTRACEAPI(3, "  H245InSetup option.");
		if (noH245inSetup)
			endPoint->DisableH245inSetup(TRUE);
		else
			endPoint->DisableH245inSetup(FALSE);
		//WRAPTRACEAPI(3, "  Bandwidth limit option.");
		//endPoint->SetInitialBandwidth(bwLimit);

		endPoint->DisableDetectInBandDTMF(TRUE);
		/*if (noInBandDTMF) {
			endPoint->DisableDetectInBandDTMF(TRUE);
			cout << " XXXXX IN BAND DTMF DISABLED XXXX" << endl;
		} else
			endPoint->DisableDetectInBandDTMF(FALSE);*/
		WRAPTRACEAPI(3, "  Jitter options.");
		endPoint->SetAudioJitterDelay(jitterMin, jitterMax);
#ifdef HAS_OH323MODS
		endPoint->SetJitterBufferAmount(jitterDec, jitterInc);
#endif

		WRAPTRACEAPI(3, "  RTP IP TOS option.");
		endPoint->SetRtpIpTypeofService(ipTos);

		Con_noFastStart = endPoint->IsFastStartDisabled();
		Con_noH245Tunnelling = endPoint->IsH245TunnelingDisabled();
		Con_noH245inSetup = endPoint->IsH245inSetupDisabled();
		/*Con_noInBandDTMF = endPoint->DetectInBandDTMFDisabled();*/
		Con_noSilenceSuppression = noSilenceSuppression;
		Con_jitterMin = jitterMin;
		Con_jitterMax = jitterMax;

		return 0;
	}

	/**************************************************************************
     *
     */
	int
	h323_set_gk(gkmode_t mode, char *gkname, char *gkpass, int gkttl,
				char **alias_tab, int alias_num)
	{
		int i;

		WRAPTRACEAPI(2, "Configuring gatekeeper.");

		if (end_point_exist() == NO)
			return -1;

		/* Set the gatekeeper password */
		if (gkpass != NULL) {
			if (strlen(gkpass)) {
				PString *gkPass = new PString(gkpass);
				endPoint->SetGatekeeperPassword(*gkPass);
			}
		}
		/* Set the gatekeeper TTL */
		endPoint->SetGatekeeperTimeToLive(gkttl);

		if (gkname == NULL)
			return 0;

		//PString *gkName = new PString(gkname);

		/* Set the local user name and our aliases */
		if ((alias_num <= 0)||(alias_tab == NULL)) {
			//PString *pstr = new PString("ASTERISK");
			PString *pstr = new PString("*");
			endPoint->SetLocalUserName(*pstr);
			delete pstr;
		} else {
			PString *pstr;
			pstr = new PString(alias_tab[0]);
			endPoint->SetLocalUserName(*pstr);
			for (i=1; i<alias_num; i++) {
				pstr = new PString(alias_tab[i]);
				endPoint->AddAliasName(*pstr);
				delete pstr;
			}
		}
		return(h323_reset_gk(mode, gkname));
	}

	int
	h323_reset_gk(gkmode_t mode, char *gkname)
	{
		if (end_point_exist() == NO)
			return(-1);

#if 0
		GKRegThread gkRegThread(mode, gkname);
		gkRegThread.SetNoAutoDelete();
		gkRegThread.Resume();
		gkRegThread.WaitForTermination();
#endif
		if (gkRegThread == NULL) {
			gkRegThread = new GKRegThread(mode, gkname);
		} else {
			gkRegThread->WaitForTermination();
			delete gkRegThread;
			gkRegThread = new GKRegThread(mode, gkname);
		}
		gkRegThread->SetNoAutoDelete();
		gkRegThread->Resume();

		return(0);
	}

	/**************************************************************************
	 * Return the Gatekeeper name (ID@hostname) we are registered with.
	 * It is returned in the user supplied buffer. If no gatekeeper is used,
	 * then  is returned. In case of success, 0 is returned and the GK
	 * name is stored in the user supplied buffer. In case of failure
	 * a negative number is returned indicating the type of failure.
	 */
	int
	h323_get_gk(char *gk, int buff_len)
	{
		WRAPTRACEAPI(4, "Checking gatekeeper.");

		if ((end_point_exist() == NO)||(gk == NULL)) {
			return(OH323GK_FAILED);
		}

		H323Gatekeeper *GK = endPoint->GetGatekeeper();
		if (GK != NULL) {
			PString gkname = GK->GetName();
			memset(gk, 0, buff_len);
			strncpy(gk, (const char*)gkname, buff_len - 1);
			if (endPoint->IsRegisteredWithGatekeeper() == FALSE) {
				return(OH323GK_NOTREGISTERED);
			}
		} else {
			return(OH323GK_NOTUSED);
		}
		return(0);
	}

	/**************************************************************************
     *
     */
	int
	h323_get_conn_info(const char *tok, char *buf, int size)
	{
		if (end_point_exist() == NO) {
			return -1;
		}
		PString token = PString(tok);
		endPoint->GetConnectionInfo(token, buf, size);
		return 0;
	}

	/**************************************************************************
     *
     */
	unsigned
	h323_check_bandwidth()
	{
		if (end_point_exist() == NO) {
			return 0;
		}
		return endPoint->GetBandwidthAvailable();
	}

	/**************************************************************************
	 * Remove the listener with type name "lis" from the list
	 * with the current listeners. 
	 */
	lis_ret_val_t
	h323_remove_listener(lis_type lis)
	{
		list<WrapListener*>::iterator	bit, cit;
		lis_ret_val_t					ret_val;

		WRAPTRACEAPI(2, "Removing listener.");

		if (end_point_exist() == NO) {
			return LIS_EP_ER;
		}

		chkLis = lis;
		cit = find_if(bit = listenerList.begin(), listenerList.end(), check_lis_type);

		/* if the iterator points to the last element and it's type is different than
		   the requested type, then the requested type is not supported */
		if (cit == listenerList.end() || !((*cit)->listenerType == lis)) {
			ret_val = LIS_NSUP_ER;
			goto exit;
		}

		/* Remove the actual listener from the endpoint */
		endPoint->RemoveListener((*cit)->listener);
		listenerList.erase(cit);
		ret_val = LIS_REMOVE_OK;

	exit:
		return ret_val;
	}

	/**************************************************************************
	 * Remove all listeners from our list.
	 */
	lis_ret_val_t
	h323_removeall_listeners()
	{
		list<WrapListener*>::iterator	lit;

		WRAPTRACEAPI(2, "Removing all listeners.");

		if (end_point_exist() == NO) {
			return LIS_EP_ER;
		}

		lit = listenerList.begin();
		while (lit != listenerList.end()) {
			WRAPTRACEAPI(2, "Removing listener " << *((*lit)->listener));
			endPoint->RemoveListener((*lit)->listener);
			listenerList.erase(lit);
			lit = listenerList.begin();
		}

		return LIS_REMOVEALL_OK;
	}

	/**************************************************************************
	 * Remove all capabilities from our list and the endpoint.
	 */
	cap_ret_val_t
	h323_removeall_capabilities()
	{
		PStringArray					str_cap;

		WRAPTRACEAPI(2, "Removing all capabilities.");
		
		if (end_point_exist() == NO) {
			return CAP_EP_ER;
		}
		endPoint->RemoveAllCapabilities();

		return CAP_REMOVEALL_OK;
	}

	/**************************************************************************
	 * Send a DTMF tone over the H323Connection with the
	 * specified token.
	 */
	void
	h323_send_tone(const char *call_token, char tone)
	{
		if (end_point_exist() == NO) {
			return;
		}

		PString token = PString(call_token);
		PString user_input = PString(tone);
		endPoint->SendUserInput(token, user_input);
	}

	/**************************************************************************
	 * Send text message over the H323Connection with the
	 * specified token.
	 */
	void
	h323_send_text(const char *call_token, char *text)
	{
		if (end_point_exist() == NO) {
			return;
		}

		PString token = PString(call_token);
		PString message = "MSG" + PString(text);
		endPoint->SendUserInput(token, message);
	}

	/**************************************************************************
	 * Make a call to the remote end point.
	 */
	call_ret_val_t 
	h323_make_call(char *host, call_details_t *pcd, call_options_t call_options)
	{
		call_ret_val_t	ret_val;
		PString			token;
		unsigned int	call_reference;
		int				i, cap_num;
		H323Capability	*cap_list[WRAP_MAX_CAP_SET+1], *cap;

		WRAPTRACEAPI(2, "Making call.");

		if (end_point_exist() == NO) {
			return CALL_EP_ER;
		}

		PString dest = PString(host);
		i = 0;
		while (call_options.cap[i] != 0) {
			cap = h323_capability_create(endPoint, 
							(cap_type_t)(call_options.cap[i]), 0);
			cap_list[i] = cap;
			i++;
		}
		cap_list[i] = NULL;
		cap_num = i;
		ret_val = endPoint->MakeCall(dest, token, &call_reference, 
						call_options.connectPort, cap_list,
						call_options.cid, call_options.cidname);
		memcpy((char *)(pcd->call_token), (const unsigned char *)token, 
				token.GetLength());
		pcd->call_reference = call_reference;
		/********
		for (i=0; i<cap_num; i++)
			delete cap_list[i];
		**********/

		return ret_val;
	}

	/**************************************************************************
	 * Clear an established call with the remote end point.
	 */
	call_ret_val_t 
	h323_clear_call(const char *call_token)
	{
		call_ret_val_t ret_val = CALL_END_ER;

		WRAPTRACEAPI(2, "Clearing call.");

		if (end_point_exist() == NO) {
			return CALL_EP_ER;
		}

		PString token = PString(call_token);
		if (endPoint->HasConnection(token)) {
#if 0
			if (!endPoint->IsConnectionCleared(token)) {
#endif

				ClearCallThread	*clearCallThread = new ClearCallThread(call_token);

				// XXX We should wait for termination and then delete
				//     the PThread object, but this causes a NULL pointer
				//     reference !!!
				//clearCallThread->Main();
				//clearCallThread->WaitForTermination();
				//delete clearCallThread;
				ret_val = CALL_END_OK;
#if 0
			} else {
				ret_val = CALL_END_OK;
				cout << " === Asked to clear a call (" 
					 << call_token << ") that has already been cleared ===" << endl;
			}
#endif
		}

		return ret_val;
	}

	/**************************************************************************
	 * Answer a pending call.
	 */
	call_ret_val_t 
	h323_answer_call(const char *call_token)
	{
		call_ret_val_t ret_val;
		//H323Capability	*cap_list[2];

		WRAPTRACEAPI(2, "Answering call.");

		if (end_point_exist() == NO) {
			return CALL_EP_ER;
		}

		PString token = PString(call_token);
		//cap_list[0] = h323_capability_create(endPoint, (cap_type_t)answer_cap, 0);
		//cap_list[1] = NULL;
		//if (endPoint->AnswerCall(token, cap_list) == FALSE)
		if (endPoint->AnswerCall(token) == FALSE)
			ret_val = CALL_ANS_ER;
		else
			ret_val = CALL_ANS_OK;

		return ret_val;
	}

	/**************************************************************************
	 * Handle indications on the specified connection.
	 */
	call_ret_val_t 
	h323_indicate_call(const char *call_token, indication_t type)
	{
		call_ret_val_t ret_val;

		WRAPTRACEAPI(2, "Sending indication " << type);

		if (end_point_exist() == NO) {
			return CALL_EP_ER;
		}

		PString token = PString(call_token);
		if (endPoint->IndicateCall(token, type) == FALSE)
			ret_val = CALL_IND_ER;
		else
			ret_val = CALL_IND_OK;

		return ret_val;
	}

	/**************************************************************************
	 * Return the status of a call.
	 */
	int
	h323_is_call_connected(const char *call_token)
	{
		WRAPTRACEAPI(2, "Checking call connection status.");

		if (end_point_exist() == NO) {
			return CALL_EP_ER;
		}

		PString token = PString(call_token);
		return (int)(endPoint->IsConnectionEstablished(token));
	}

	/**************************************************************************
	 * Get the reason of a cleared call.
	 */
	int
	h323_get_reason_code(int reason)
	{
		int i, res;

		res = -1;
		i = 0;
		while (call_end_reason[i].h323_reason != -1) {
			res = call_end_reason[i].wrap_reason;
			if (reason == call_end_reason[i].h323_reason)
				break;
			i++;
		}
		return(res);
	}

	/**************************************************************************
	 * Get the description of the reason of a cleared call.
	 */
	char *
	h323_get_reason_desc(int reason)
	{
		int i;
		char *res;

		res = NULL;
		i = 0;
		while (call_end_reason[i].h323_reason != -1) {
			res = call_end_reason[i].desc;
			if (reason == call_end_reason[i].h323_reason)
				break;
			i++;
		}
		return(res);
	}

	/**************************************************************************
	 * Change the mode of a running call.
	 */
	call_ret_val_t 
	h323_change_call(const char *call_token, const char *new_mode)
	{
		call_ret_val_t ret_val;

		WRAPTRACEAPI(2, "Changing call.");

		if (end_point_exist() == NO) {
			return CALL_EP_ER;
		}

		PString token = PString(call_token);
		PString mode = PString(new_mode);
		if (endPoint->ChangeMode(token, mode) != TRUE)
			ret_val = CALL_CHG_OK;
		else
			ret_val = CALL_CHG_ER;

		return ret_val;
	}

} /* extern "C" */

// End of file //////////////////////////////////////////////////////////////
